// 简单推特关注机器人 - 直接关注方案
class SimpleFollowBot {
    constructor() {
        this.isRunning = false;
        this.stats = {
            tweetsViewed: 0,
            followsGiven: 0,
            startTime: Date.now()
        };
        this.targets = {
            tweetsToView: 30 + Math.floor(Math.random() * 21), // 30-50条
            followsToGive: 3 + Math.floor(Math.random() * 3)   // 3-5个
        };
    }

    // 延迟函数
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    // 带倒计时的延迟
    async delayWithCountdown(ms, message) {
        const startTime = Date.now();
        const endTime = startTime + ms;
        
        while (Date.now() < endTime) {
            const remaining = Math.ceil((endTime - Date.now()) / 1000);
            if (remaining > 0) {
                console.log(`⏳ ${message} ${remaining}秒...`);
            }
            await this.delay(1000);
        }
        console.log(`✅ ${message} 完成`);
    }

    // 模拟人类点击
    async simulateHumanClick(element) {
        const rect = element.getBoundingClientRect();
        const x = rect.left + rect.width / 2;
        const y = rect.top + rect.height / 2;
        
        const clickEvent = new MouseEvent('click', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(clickEvent);
    }

    // 方法1: 直接查找关注按钮（在主页）
    async findFollowButtonOnHome() {
        console.log('🔍 在主页查找关注按钮...');
        
        // 多种关注按钮选择器
        const followSelectors = [
            'button[data-testid="follow"]',
            'div[role="button"][data-testid="follow"]',
            'button:contains("关注")',
            'div[role="button"]:contains("关注")',
            'button:contains("Follow")',
            'div[role="button"]:contains("Follow")',
            'button[aria-label*="关注"]',
            'div[role="button"][aria-label*="关注"]',
            'button[aria-label*="Follow"]',
            'div[role="button"][aria-label*="Follow"]'
        ];
        
        for (const selector of followSelectors) {
            try {
                const buttons = document.querySelectorAll(selector);
                console.log(`🔍 选择器 "${selector}" 找到 ${buttons.length} 个按钮`);
                
                for (const button of buttons) {
                    if (button.offsetParent !== null) {
                        const text = button.textContent?.trim() || '';
                        const ariaLabel = button.getAttribute('aria-label') || '';
                        
                        console.log(`🔍 检查按钮: text="${text}", aria-label="${ariaLabel}"`);
                        
                        if (text.includes('关注') || text.includes('Follow') || 
                            ariaLabel.includes('关注') || ariaLabel.includes('Follow')) {
                            console.log(`✅ 找到关注按钮: ${selector}`);
                            return button;
                        }
                    }
                }
            } catch (e) {
                console.log(`❌ 选择器 "${selector}" 出错:`, e.message);
                continue;
            }
        }
        
        // 广泛搜索所有按钮
        console.log('🔍 使用广泛搜索策略...');
        const allButtons = document.querySelectorAll('button, div[role="button"]');
        console.log(`🔍 页面总共有 ${allButtons.length} 个按钮`);
        
        for (const button of allButtons) {
            const text = button.textContent?.trim() || '';
            const ariaLabel = button.getAttribute('aria-label') || '';
            
            if ((text === '关注' || text === 'Follow' || 
                 ariaLabel.includes('关注') || ariaLabel.includes('Follow')) &&
                button.offsetParent !== null) {
                console.log(`✅ 广泛搜索找到关注按钮: text="${text}", aria-label="${ariaLabel}"`);
                return button;
            }
        }
        
        return null;
    }

    // 方法2: 通过用户名搜索并关注
    async searchAndFollow(username) {
        try {
            console.log(`🔍 搜索用户: @${username}`);
            
            // 导航到搜索页面
            const searchUrl = `https://x.com/search?q=${encodeURIComponent(username)}&src=typed_query&f=user`;
            window.location.href = searchUrl;
            
            await this.delayWithCountdown(3000, '等待搜索结果加载');
            
            // 查找用户卡片
            const userCards = document.querySelectorAll('[data-testid="UserCell"]');
            console.log(`🔍 找到 ${userCards.length} 个用户卡片`);
            
            if (userCards.length > 0) {
                const userCard = userCards[0];
                console.log('🎯 点击第一个用户卡片');
                
                // 点击用户卡片进入主页
                await this.simulateHumanClick(userCard);
                await this.delayWithCountdown(2000, '等待用户主页加载');
                
                // 在用户主页查找关注按钮
                const followButton = await this.findFollowButtonOnHome();
                if (followButton) {
                    await this.simulateHumanClick(followButton);
                    console.log('✅ 已点击关注按钮');
                    this.stats.followsGiven++;
                    return true;
                }
            }
            
            return false;
        } catch (error) {
            console.error('搜索关注失败:', error);
            return false;
        }
    }

    // 方法3: 通过推荐用户关注
    async followRecommendedUsers() {
        try {
            console.log('🔍 查找推荐用户...');
            
            // 查找推荐用户区域
            const recommendedSelectors = [
                '[data-testid="UserCell"]',
                '[data-testid="UserRecommendation"]',
                '[data-testid="UserToFollow"]',
                'div[role="button"][data-testid="UserCell"]'
            ];
            
            let userElements = [];
            for (const selector of recommendedSelectors) {
                const elements = document.querySelectorAll(selector);
                userElements = userElements.concat(Array.from(elements));
            }
            
            console.log(`🔍 找到 ${userElements.length} 个推荐用户`);
            
            if (userElements.length > 0) {
                const randomUser = userElements[Math.floor(Math.random() * userElements.length)];
                console.log('🎯 选择推荐用户');
                
                // 点击用户进入主页
                await this.simulateHumanClick(randomUser);
                await this.delayWithCountdown(2000, '等待用户主页加载');
                
                // 查找关注按钮
                const followButton = await this.findFollowButtonOnHome();
                if (followButton) {
                    await this.simulateHumanClick(followButton);
                    console.log('✅ 已关注推荐用户');
                    this.stats.followsGiven++;
                    return true;
                }
            }
            
            return false;
        } catch (error) {
            console.error('关注推荐用户失败:', error);
            return false;
        }
    }

    // 方法4: 通过时间线用户关注
    async followTimelineUsers() {
        try {
            console.log('🔍 在时间线查找用户...');
            
            // 查找时间线中的用户头像
            const avatarSelectors = [
                '[data-testid="UserAvatar-Container"]',
                '[data-testid="UserAvatar"]',
                'article [data-testid="UserAvatar-Container"]',
                'article [data-testid="UserAvatar"]'
            ];
            
            let avatars = [];
            for (const selector of avatarSelectors) {
                const elements = document.querySelectorAll(selector);
                avatars = avatars.concat(Array.from(elements));
            }
            
            // 过滤可见的头像
            const visibleAvatars = avatars.filter(avatar => {
                const isVisible = avatar.offsetParent !== null;
                const hasSize = avatar.offsetWidth > 0 && avatar.offsetHeight > 0;
                const isInTweet = avatar.closest('article') || avatar.closest('[data-testid="tweet"]');
                return isVisible && hasSize && isInTweet;
            });
            
            console.log(`🔍 找到 ${visibleAvatars.length} 个可见头像`);
            
            if (visibleAvatars.length > 0) {
                const randomAvatar = visibleAvatars[Math.floor(Math.random() * visibleAvatars.length)];
                console.log('🎯 选择时间线用户头像');
                
                // 点击头像进入主页
                await this.simulateHumanClick(randomAvatar);
                await this.delayWithCountdown(2000, '等待用户主页加载');
                
                // 查找关注按钮
                const followButton = await this.findFollowButtonOnHome();
                if (followButton) {
                    await this.simulateHumanClick(followButton);
                    console.log('✅ 已关注时间线用户');
                    this.stats.followsGiven++;
                    return true;
                }
            }
            
            return false;
        } catch (error) {
            console.error('关注时间线用户失败:', error);
            return false;
        }
    }

    // 自然浏览
    async scrollAndBrowse() {
        try {
            const scrollDistance = Math.random() * 300 + 300;
            
            window.scrollBy({
                top: scrollDistance,
                behavior: 'smooth'
            });
            
            console.log(`📜 向下滑动 ${Math.round(scrollDistance)}px，浏览内容...`);
            this.stats.tweetsViewed++;
            
            const isInterested = Math.random() < 0.3;
            
            if (isInterested) {
                const readingTime = Math.random() * 4 + 2;
                console.log(`👀 发现感兴趣的内容！阅读时间: ${readingTime.toFixed(1)}秒`);
                await this.delayWithCountdown(readingTime * 1000, '阅读感兴趣的内容');
            } else {
                const skipTime = Math.random() * 1.5 + 0.5;
                console.log(`⏭️ 内容不感兴趣，快速跳过 (${skipTime.toFixed(1)}秒)`);
                await this.delayWithCountdown(skipTime * 1000, '快速跳过不感兴趣内容');
            }
            
            return true;
        } catch (error) {
            console.error('自然浏览失败:', error);
            return false;
        }
    }

    // 显示进度
    showProgress() {
        const elapsed = Math.round((Date.now() - this.stats.startTime) / 1000);
        console.log(`📊 进度: ${elapsed}秒 | 推文:${this.stats.tweetsViewed}/${this.targets.tweetsToView} | 关注:${this.stats.followsGiven}/${this.targets.followsToGive}`);
    }

    // 开始关注流程
    async startFollowProcess() {
        if (this.isRunning) {
            console.log('⚠️ 关注流程已在进行中');
            return;
        }
        
        this.isRunning = true;
        console.log('🚀 开始简单关注流程...');
        console.log(`🎯 目标: 浏览${this.targets.tweetsToView}条推文, 关注${this.targets.followsToGive}个用户`);
        
        const startTime = Date.now();
        
        while (this.stats.followsGiven < this.targets.followsToGive && this.isRunning) {
            // 随机选择关注方法
            const methods = [
                () => this.followTimelineUsers(),
                () => this.followRecommendedUsers(),
                () => this.searchAndFollow('liuruyy'), // 可以替换为其他用户名
                () => this.searchAndFollow('randomuser')
            ];
            
            const randomMethod = methods[Math.floor(Math.random() * methods.length)];
            
            try {
                console.log('🔄 尝试关注用户...');
                const success = await randomMethod();
                
                if (success) {
                    console.log('✅ 关注成功！');
                    await this.delayWithCountdown(3000, '关注完成，等待继续');
                } else {
                    console.log('❌ 关注失败，尝试浏览内容...');
                    await this.scrollAndBrowse();
                    await this.delayWithCountdown(2000, '浏览完成，等待继续');
                }
                
                this.showProgress();
                
            } catch (error) {
                console.error('关注流程出错:', error);
                await this.delay(2000);
            }
        }
        
        console.log('✅ 关注流程完成!');
        console.log(`📈 最终统计: 浏览${this.stats.tweetsViewed}条推文, 关注${this.stats.followsGiven}个用户`);
        this.isRunning = false;
    }

    // 停止关注流程
    stopFollowProcess() {
        this.isRunning = false;
        console.log('⏹️ 关注流程已停止');
    }
}

// 创建实例
const followBot = new SimpleFollowBot();

// 全局控制
window.simpleFollowBot = {
    start: () => followBot.startFollowProcess(),
    stop: () => followBot.stopFollowProcess(),
    status: () => console.log('📊 状态:', {
        isRunning: followBot.isRunning,
        stats: followBot.stats
    })
};

console.log('🤖 简单关注机器人已加载');
console.log('📋 使用方法:');
console.log('   simpleFollowBot.start() - 开始关注流程');
console.log('   simpleFollowBot.stop() - 停止关注流程');
console.log('   simpleFollowBot.status() - 查看状态');

// 自动开始
followBot.startFollowProcess();
