// 简化版Twitter互动机器人 - 绕过扩展通信问题
// 在Twitter页面控制台运行此脚本

class SimpleTwitterBot {
    constructor() {
        this.isRunning = false;
        this.stats = {
            likes: 0,
            comments: 0,
            retweets: 0,
            follows: 0
        };
        this.targetStats = {
            likes: { min: 6, max: 10 },
            comments: { min: 2, max: 4 },
            retweets: { min: 1, max: 2 }
        };
    }

    // 延迟函数
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    // 带倒计时的延迟
    async delayWithCountdown(ms, message = '等待') {
        const startTime = Date.now();
        const endTime = startTime + ms;
        
        const countdown = setInterval(() => {
            const remaining = Math.ceil((endTime - Date.now()) / 1000);
            if (remaining > 0) {
                console.log(`⏳ ${message} ${remaining}秒...`);
            }
        }, 1000);
        
        await this.delay(ms);
        clearInterval(countdown);
        console.log(`✅ ${message} 完成`);
    }

    // 模拟人类点击
    async simulateHumanClick(element) {
        if (!element) return false;
        
        // 滚动到元素可见
        element.scrollIntoView({ behavior: 'smooth', block: 'center' });
        await this.delay(500);
        
        // 模拟鼠标悬停
        element.dispatchEvent(new MouseEvent('mouseenter', { bubbles: true }));
        await this.delay(200);
        
        // 点击
        element.click();
        await this.delay(300);
        
        return true;
    }

    // 模拟人类输入
    async simulateHumanType(element, text) {
        if (!element) return false;
        
        element.focus();
        await this.delay(200);
        
        // 清空现有内容
        element.value = '';
        element.dispatchEvent(new Event('input', { bubbles: true }));
        
        // 逐字符输入
        for (let i = 0; i < text.length; i++) {
            element.value += text[i];
            element.dispatchEvent(new Event('input', { bubbles: true }));
            await this.delay(50 + Math.random() * 100);
        }
        
        // 触发change事件
        element.dispatchEvent(new Event('change', { bubbles: true }));
        await this.delay(200);
        
        return true;
    }

    // 检查元素是否在视窗内
    isElementInViewport(element) {
        const rect = element.getBoundingClientRect();
        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
            rect.right <= (window.innerWidth || document.documentElement.clientWidth)
        );
    }

    // 生成备用评论
    generateFallbackComment() {
        const fallbackComments = [
            '很有见地的观点！',
            '这个角度很有意思',
            '说得很有道理',
            '值得深思',
            '赞同这个看法',
            '很有启发性',
            '这个观点很新颖',
            '说得对',
            '很有道理',
            '值得关注',
            '这个想法不错',
            '学到了',
            '有道理',
            '确实如此',
            '同意这个观点',
            '很有意思',
            '说得很好',
            '很有道理',
            '赞同',
            '支持',
            '说得对',
            '很有道理',
            '赞同',
            '支持',
            '说得对',
            '很有道理',
            '赞同',
            '支持'
        ];
        
        return fallbackComments[Math.floor(Math.random() * fallbackComments.length)];
    }

    // 智能点赞
    async smartLike() {
        try {
            console.log('🔍 开始智能点赞...');
            
            // 查找点赞按钮
            const likeSelectors = [
                '[data-testid="like"]',
                '[aria-label*="Like"]',
                '[aria-label*="喜欢"]',
                'div[role="button"][aria-label*="Like"]',
                'div[role="button"][aria-label*="喜欢"]'
            ];
            
            let likeButton = null;
            for (const selector of likeSelectors) {
                const buttons = document.querySelectorAll(selector);
                for (const button of buttons) {
                    if (this.isElementInViewport(button) && button.offsetParent !== null) {
                        likeButton = button;
                        break;
                    }
                }
                if (likeButton) break;
            }
            
            if (!likeButton) {
                console.log('❌ 未找到可点赞的内容');
                return false;
            }
            
            // 检查是否已经点赞
            const isLiked = likeButton.getAttribute('data-testid') === 'unlike' || 
                          likeButton.getAttribute('aria-pressed') === 'true' ||
                          likeButton.querySelector('[data-testid="unlike"]');
            
            if (isLiked) {
                console.log('ℹ️ 内容已点赞，跳过');
                return false;
            }
            
            console.log('👆 点击点赞按钮...');
            await this.simulateHumanClick(likeButton);
            await this.delay(1000);
            
            console.log('✅ 点赞成功');
            this.stats.likes++;
            return true;
            
        } catch (error) {
            console.error('❌ 点赞失败:', error);
            return false;
        }
    }

    // 智能评论（简化版）
    async smartComment() {
        try {
            console.log('🔍 开始智能评论...');
            
            // 查找回复按钮
            const replySelectors = [
                '[data-testid="reply"]',
                '[aria-label*="Reply"]',
                '[aria-label*="回复"]',
                'div[role="button"][aria-label*="Reply"]',
                'div[role="button"][aria-label*="回复"]'
            ];
            
            let replyButton = null;
            for (const selector of replySelectors) {
                const buttons = document.querySelectorAll(selector);
                for (const button of buttons) {
                    if (this.isElementInViewport(button) && button.offsetParent !== null) {
                        replyButton = button;
                        break;
                    }
                }
                if (replyButton) break;
            }
            
            if (!replyButton) {
                console.log('❌ 未找到回复按钮');
                return false;
            }
            
            console.log('👆 点击回复按钮...');
            await this.simulateHumanClick(replyButton);
            await this.delay(2000);
            
            // 等待回复框加载
            console.log('⏳ 等待回复框加载...');
            await this.delayWithCountdown(3000, '等待回复框加载');
            
            // 查找回复输入框
            const replyInputSelectors = [
                '[data-testid="tweetTextarea_0"]',
                '[data-testid="tweetTextarea_1"]',
                '[data-testid="tweetTextarea_2"]',
                'div[data-testid="tweetTextarea_0"]',
                'div[data-testid="tweetTextarea_1"]',
                'div[data-testid="tweetTextarea_2"]',
                '[contenteditable="true"][data-testid*="tweetTextarea"]',
                'div[contenteditable="true"][role="textbox"]'
            ];
            
            let replyInput = null;
            for (const selector of replyInputSelectors) {
                const inputs = document.querySelectorAll(selector);
                for (const input of inputs) {
                    if (this.isElementInViewport(input) && input.offsetParent !== null) {
                        replyInput = input;
                        break;
                    }
                }
                if (replyInput) break;
            }
            
            if (!replyInput) {
                console.log('❌ 未找到回复输入框');
                return false;
            }
            
            console.log('✅ 找到回复输入框');
            
            // 生成评论
            console.log('🤖 生成评论...');
            const comment = this.generateFallbackComment();
            console.log(`💭 生成的评论: "${comment}"`);
            
            // 输入评论
            console.log('⌨️ 输入评论...');
            await this.simulateHumanType(replyInput, comment);
            await this.delay(1000);
            
            // 等待输入完成
            await this.delay(2000);
            
            // 查找发送按钮
            const sendSelectors = [
                '[data-testid="tweetButton"]',
                '[data-testid="tweetButtonInline"]',
                'div[data-testid="tweetButton"]',
                'div[data-testid="tweetButtonInline"]',
                '[role="button"][data-testid*="tweetButton"]'
            ];
            
            let sendButton = null;
            for (const selector of sendSelectors) {
                const buttons = document.querySelectorAll(selector);
                for (const button of buttons) {
                    if (this.isElementInViewport(button) && button.offsetParent !== null) {
                        sendButton = button;
                        break;
                    }
                }
                if (sendButton) break;
            }
            
            if (!sendButton) {
                console.log('❌ 未找到发送按钮');
                return false;
            }
            
            // 强制激活发送按钮
            console.log('🔄 强制激活发送按钮...');
            sendButton.focus();
            await this.delay(500);
            
            // 点击发送按钮
            console.log('📤 发送评论...');
            await this.simulateHumanClick(sendButton);
            await this.delay(2000);
            
            console.log('✅ 评论成功');
            this.stats.comments++;
            return true;
            
        } catch (error) {
            console.error('❌ 评论失败:', error);
            return false;
        }
    }

    // 智能转发
    async smartRetweet() {
        try {
            console.log('🔍 开始智能转发...');
            
            // 查找转发按钮
            const retweetSelectors = [
                '[data-testid="retweet"]',
                '[aria-label*="Retweet"]',
                '[aria-label*="转发"]',
                'div[role="button"][aria-label*="Retweet"]',
                'div[role="button"][aria-label*="转发"]'
            ];
            
            let retweetButton = null;
            for (const selector of retweetSelectors) {
                const buttons = document.querySelectorAll(selector);
                for (const button of buttons) {
                    if (this.isElementInViewport(button) && button.offsetParent !== null) {
                        retweetButton = button;
                        break;
                    }
                }
                if (retweetButton) break;
            }
            
            if (!retweetButton) {
                console.log('❌ 未找到转发按钮');
                return false;
            }
            
            // 检查是否已经转发
            const isRetweeted = retweetButton.getAttribute('data-testid') === 'unretweet' || 
                              retweetButton.getAttribute('aria-pressed') === 'true' ||
                              retweetButton.querySelector('[data-testid="unretweet"]');
            
            if (isRetweeted) {
                console.log('ℹ️ 内容已转发，跳过');
                return false;
            }
            
            console.log('👆 点击转发按钮...');
            await this.simulateHumanClick(retweetButton);
            await this.delay(1000);
            
            // 查找确认转发按钮
            const confirmSelectors = [
                '[data-testid="retweetConfirm"]',
                'div[data-testid="retweetConfirm"]',
                '[role="button"][data-testid*="retweetConfirm"]'
            ];
            
            let confirmButton = null;
            for (const selector of confirmSelectors) {
                const buttons = document.querySelectorAll(selector);
                for (const button of buttons) {
                    if (this.isElementInViewport(button) && button.offsetParent !== null) {
                        confirmButton = button;
                        break;
                    }
                }
                if (confirmButton) break;
            }
            
            if (confirmButton) {
                console.log('👆 确认转发...');
                await this.simulateHumanClick(confirmButton);
                await this.delay(1000);
            }
            
            console.log('✅ 转发成功');
            this.stats.retweets++;
            return true;
            
        } catch (error) {
            console.error('❌ 转发失败:', error);
            return false;
        }
    }

    // 自然浏览
    async naturalBrowsing() {
        try {
            console.log('🔍 开始自然浏览...');
            
            // 随机滚动
            const scrollAmount = Math.random() * 500 + 200;
            window.scrollBy(0, scrollAmount);
            await this.delay(1000 + Math.random() * 2000);
            
            // 随机停留
            const stayTime = Math.random() * 3000 + 1000;
            await this.delay(stayTime);
            
            return true;
        } catch (error) {
            console.error('❌ 自然浏览失败:', error);
            return false;
        }
    }

    // 选择下一个动作
    selectNextAction() {
        const actions = [];
        
        // 根据目标统计添加动作
        if (this.stats.likes < this.targetStats.likes.max) {
            actions.push('like');
        }
        if (this.stats.comments < this.targetStats.comments.max) {
            actions.push('comment');
        }
        if (this.stats.retweets < this.targetStats.retweets.max) {
            actions.push('retweet');
        }
        
        // 总是包含浏览动作
        actions.push('browse');
        
        if (actions.length === 0) {
            return 'browse';
        }
        
        // 随机选择动作，但给浏览更高权重
        const weights = {
            'like': 0.3,
            'comment': 0.2,
            'retweet': 0.1,
            'browse': 0.4
        };
        
        const random = Math.random();
        let cumulative = 0;
        
        for (const action of actions) {
            cumulative += weights[action] || 0.1;
            if (random <= cumulative) {
                return action;
            }
        }
        
        return actions[Math.floor(Math.random() * actions.length)];
    }

    // 显示进度
    showProgress() {
        const total = this.targetStats.likes.max + this.targetStats.comments.max + this.targetStats.retweets.max;
        const completed = this.stats.likes + this.stats.comments + this.stats.retweets;
        const progress = Math.round((completed / total) * 100);
        
        console.log(`📊 互动进度: ${progress}% | 点赞:${this.stats.likes}/${this.targetStats.likes.max} | 评论:${this.stats.comments}/${this.targetStats.comments.max} | 转发:${this.stats.retweets}/${this.targetStats.retweets.max}`);
    }

    // 显示最终统计
    showFinalStats() {
        console.log('🎉 互动完成！最终统计：');
        console.log(`👍 点赞: ${this.stats.likes} 次`);
        console.log(`💬 评论: ${this.stats.comments} 次`);
        console.log(`🔄 转发: ${this.stats.retweets} 次`);
        console.log(`📈 总计: ${this.stats.likes + this.stats.comments + this.stats.retweets} 次互动`);
    }

    // 停止互动
    stopInteraction() {
        this.isRunning = false;
        console.log('⏹️ 互动已停止');
        this.showFinalStats();
    }

    // 开始互动
    async startInteraction() {
        if (this.isRunning) {
            console.log('⚠️ 互动已在进行中');
            return;
        }

        console.log('🚀 开始Twitter互动机器人（简化版）');
        console.log('🎯 目标：用高质量互动提高账号权重');
        console.log('📊 计划：点赞 6-10 条，评论 2-4 条，转发 1-2 条');
        console.log('⏰ 预计时间：10分钟');
        console.log('💡 使用备用评论系统，无需扩展');
        
        this.isRunning = true;
        this.stats = { likes: 0, comments: 0, retweets: 0, follows: 0 };
        
        const startTime = Date.now();
        const duration = 10 * 60 * 1000; // 10分钟
        
        try {
            while (this.isRunning && (Date.now() - startTime) < duration) {
                const action = this.selectNextAction();
                console.log(`🎯 执行动作: ${action}`);
                
                let success = false;
                switch (action) {
                    case 'like':
                        success = await this.smartLike();
                        break;
                    case 'comment':
                        success = await this.smartComment();
                        break;
                    case 'retweet':
                        success = await this.smartRetweet();
                        break;
                    case 'browse':
                        success = await this.naturalBrowsing();
                        break;
                }
                
                if (success) {
                    console.log(`✅ ${action} 执行成功`);
                } else {
                    console.log(`❌ ${action} 执行失败`);
                }
                
                this.showProgress();
                
                // 随机等待时间
                const waitTime = Math.random() * 10000 + 5000; // 5-15秒
                console.log(`⏳ 互动等待 ${Math.round(waitTime/1000)}秒...`);
                await this.delay(waitTime);
            }
            
            if (this.isRunning) {
                console.log('⏰ 时间到，互动结束');
            }
            
        } catch (error) {
            console.error('❌ 互动过程中出错:', error);
        } finally {
            this.stopInteraction();
        }
    }
}

// 创建机器人实例
const bot = new SimpleTwitterBot();

// 开始互动
console.log('🤖 Twitter互动机器人（简化版）已准备就绪');
console.log('💡 输入 bot.startInteraction() 开始互动');
console.log('💡 输入 bot.stopInteraction() 停止互动');
console.log('💡 输入 bot.showProgress() 查看进度');

// 自动开始（可选）
// bot.startInteraction();
