// 推特养号助手 - 弹窗脚本
class PopupController {
    constructor() {
        this.isRunning = false;
        this.config = {};
        this.stats = {};
        this.actionLog = [];
        this.timerInterval = null;
        this.lastStatsUpdate = 0;

        // 缓存常用的DOM元素
        this.statusElement = document.getElementById('status');
        this.statusTextElement = document.getElementById('statusText');
        this.progressFillElement = document.getElementById('progressFill');
        this.statsContainer = document.getElementById('stats');
        this.timerPanel = document.getElementById('timerPanel');
        this.remainingTimeElement = document.getElementById('remainingTime');
        this.elapsedTimeElement = document.getElementById('elapsedTime');
        this.stageTextElement = document.getElementById('stageText');
        this.logSection = document.getElementById('logSection');
        this.actionListElement = document.getElementById('actionList');
        this.emptyLogElement = document.getElementById('emptyLog');
        this.startButton = document.getElementById('startBtn');
        this.stopButton = document.getElementById('stopBtn');

        this.init();
    }

    async init() {
        // 绑定事件监听器
        this.bindEvents();
        this.bindRuntimeMessages();
        
        // 加载配置
        await this.loadConfig();
        
        // 检查状态
        await this.checkStatus();
        
        // 开始状态更新循环
        this.startStatusUpdate();
        
        console.log('🤖 弹窗控制器已初始化');
    }

    bindRuntimeMessages() {
        chrome.runtime.onMessage.addListener((message) => {
            if (!message) return;
            if (message.action === 'botBroadcast' && message.stats) {
                this.updateStats(message.stats);
                this.updateStatusUI();
            }
        });
    }

    bindEvents() {
        // 开始按钮
        document.getElementById('startBtn').addEventListener('click', () => {
            this.startBot();
        });

        // 停止按钮
        document.getElementById('stopBtn').addEventListener('click', () => {
            this.stopBot();
        });

        // 保存配置按钮
        document.getElementById('saveBtn').addEventListener('click', () => {
            this.saveConfig();
        });

        // 配置变化时自动保存
        const configInputs = [
            'tweetsToView', 'duration', 'likesToGive', 'followsToGive', 'retweetsToGive',
            'enableScroll', 'enableLike', 'enableFollow', 'enableRetweet'
        ];

        configInputs.forEach(id => {
            const element = document.getElementById(id);
            if (element) {
                element.addEventListener('change', () => {
                    this.saveConfig();
                });
            }
        });
    }

    async loadConfig() {
        try {
            const result = await chrome.storage.sync.get([
                'tweetsToView', 'duration', 'likesToGive', 'followsToGive', 'retweetsToGive',
                'enableScroll', 'enableLike', 'enableFollow', 'enableRetweet'
            ]);

            this.config = {
                tweetsToView: result.tweetsToView || 60,
                duration: result.duration || 20,
                likesToGive: result.likesToGive || 5,
                followsToGive: result.followsToGive || 3,
                retweetsToGive: result.retweetsToGive || 2,
                enableScroll: result.enableScroll !== false,
                enableLike: result.enableLike !== false,
                enableFollow: result.enableFollow !== false,
                enableRetweet: result.enableRetweet !== false
            };

            this.updateUI();
            console.log('📋 配置已加载:', this.config);
        } catch (error) {
            console.error('❌ 加载配置失败:', error);
        }
    }

    async saveConfig() {
        try {
            const config = {
                tweetsToView: parseInt(document.getElementById('tweetsToView').value),
                duration: parseInt(document.getElementById('duration').value),
                likesToGive: parseInt(document.getElementById('likesToGive').value),
                followsToGive: parseInt(document.getElementById('followsToGive').value),
                retweetsToGive: parseInt(document.getElementById('retweetsToGive').value),
                enableScroll: document.getElementById('enableScroll').checked,
                enableLike: document.getElementById('enableLike').checked,
                enableFollow: document.getElementById('enableFollow').checked,
                enableRetweet: document.getElementById('enableRetweet').checked
            };

            await chrome.storage.sync.set(config);
            await chrome.runtime.sendMessage({ action: 'updateConfig', config });
            this.config = config;
            
            // 显示保存成功提示
            this.showMessage('配置已保存', 'success');
            console.log('💾 配置已保存:', config);
        } catch (error) {
            console.error('❌ 保存配置失败:', error);
            this.showMessage('保存配置失败', 'error');
        }
    }

    updateUI() {
        // 更新输入框
        document.getElementById('tweetsToView').value = this.config.tweetsToView;
        document.getElementById('duration').value = this.config.duration;
        document.getElementById('likesToGive').value = this.config.likesToGive;
        document.getElementById('followsToGive').value = this.config.followsToGive;
        document.getElementById('retweetsToGive').value = this.config.retweetsToGive;
        
        // 更新复选框
        document.getElementById('enableScroll').checked = this.config.enableScroll;
        document.getElementById('enableLike').checked = this.config.enableLike;
        document.getElementById('enableFollow').checked = this.config.enableFollow;
        document.getElementById('enableRetweet').checked = this.config.enableRetweet;
    }

    async checkStatus() {
        try {
            const response = await chrome.runtime.sendMessage({ action: 'status' });
            
            if (response && response.success) {
                this.isRunning = response.isRunning;

                if (response.config) {
                    this.config = response.config;
                    if (!this.isRunning) {
                        this.updateUI();
                    }
                }

                if (response.stats) {
                    this.updateStats(response.stats);
                }

                this.updateStatusUI();
            }
        } catch (error) {
            console.error('❌ 检查状态失败:', error);
        }
    }

    async startBot() {
        try {
            // 先保存当前配置
            await this.saveConfig();
            
            // 发送启动消息，让后台脚本处理标签页获取
            const response = await chrome.runtime.sendMessage({ action: 'start' });
            
            if (response && response.success) {
                this.isRunning = true;
                this.updateStats({
                    tweetsViewed: 0,
                    likesGiven: 0,
                    followsGiven: 0,
                    retweetsGiven: 0,
                    targets: {
                        likesToGive: this.config.likesToGive,
                        followsToGive: this.config.followsToGive,
                        retweetsToGive: this.config.retweetsToGive
                    },
                    totalDurationSeconds: this.config.duration * 60,
                    remainingSeconds: this.config.duration * 60,
                    elapsedSeconds: 0,
                    recentActions: [],
                    stage: 'start'
                });
                this.updateStatusUI();
                this.showMessage('机器人已启动', 'success');
            } else {
                this.showMessage(response?.message || '启动失败', 'error');
            }
        } catch (error) {
            console.error('❌ 启动机器人失败:', error);
            this.showMessage('启动失败: ' + error.message, 'error');
        }
    }

    async stopBot() {
        try {
            const response = await chrome.runtime.sendMessage({ action: 'stop' });
            
            if (response.success) {
                this.isRunning = false;
                this.updateStatusUI();
                this.showMessage('机器人已停止', 'success');
            } else {
                this.showMessage('停止失败', 'error');
            }
        } catch (error) {
            console.error('❌ 停止机器人失败:', error);
            this.showMessage('停止失败', 'error');
        }
    }

    updateStatusUI() {
        const hasStats = this.stats && Object.keys(this.stats).length > 0;
        const stageLabel = this.getStageDescription(this.stats?.stage, this.isRunning);

        if (this.statusElement) {
            this.statusElement.className = `status ${this.isRunning ? 'running' : 'stopped'}`;
        }

        if (this.statusTextElement) {
            this.statusTextElement.textContent = stageLabel || (this.isRunning ? '正在运行' : '已停止');
        }

        if (this.startButton) {
            this.startButton.disabled = this.isRunning;
        }
        if (this.stopButton) {
            this.stopButton.disabled = !this.isRunning;
        }

        if (this.statsContainer) {
            this.statsContainer.style.display = (this.isRunning || hasStats) ? 'grid' : 'none';
        }

        if (!hasStats && this.logSection) {
            this.logSection.style.display = this.isRunning ? 'block' : 'none';
        }

        this.updateTimerDisplay();
    }

    updateStats(stats) {
        if (!stats) {
            this.stats = {};
            this.actionLog = [];
            document.getElementById('tweetsViewed').textContent = '0';
            document.getElementById('likesGiven').textContent = '0';
            document.getElementById('followsGiven').textContent = '0';
            document.getElementById('retweetsGiven').textContent = '0';
            if (this.progressFillElement) {
                this.progressFillElement.style.width = '0%';
            }
            if (this.timerPanel) {
                this.timerPanel.classList.add('hidden');
            }
            if (this.stageTextElement) {
                this.stageTextElement.classList.add('hidden');
            }
            if (this.logSection) {
                this.logSection.style.display = 'none';
            }
            return;
        }

        this.stats = stats;
        this.lastStatsUpdate = Date.now();
        this.ensureTimerTick();
        this.actionLog = stats.recentActions || [];

        if (typeof stats.isRunning === 'boolean') {
            this.isRunning = stats.isRunning;
        }

        document.getElementById('tweetsViewed').textContent = stats.tweetsViewed ?? 0;
        document.getElementById('likesGiven').textContent = stats.likesGiven ?? 0;
        document.getElementById('followsGiven').textContent = stats.followsGiven ?? 0;
        document.getElementById('retweetsGiven').textContent = stats.retweetsGiven ?? 0;
        
        const targetLikes = stats.targets?.likesToGive ?? this.config.likesToGive ?? 0;
        const targetFollows = stats.targets?.followsToGive ?? this.config.followsToGive ?? 0;
        const targetRetweets = stats.targets?.retweetsToGive ?? this.config.retweetsToGive ?? 0;
        const totalInteractions = targetLikes + targetFollows + targetRetweets;
        const completedInteractions = (stats.likesGiven ?? 0) + (stats.followsGiven ?? 0) + (stats.retweetsGiven ?? 0);
        const progress = stats.interactionProgress ?? (totalInteractions > 0 ? (completedInteractions / totalInteractions) * 100 : 0);
        
        if (this.progressFillElement) {
            this.progressFillElement.style.width = `${Math.min(progress, 100)}%`;
        }

        if (this.logSection) {
            this.logSection.style.display = 'block';
        }

        this.renderActionLog(this.actionLog);
        this.updateTimerDisplay();
    }

    ensureTimerTick() {
        if (this.timerInterval) {
            return;
        }
        this.timerInterval = setInterval(() => {
            if (this.stats && Object.keys(this.stats).length > 0) {
                this.updateTimerDisplay();
            }
        }, 1000);
    }

    updateTimerDisplay() {
        if (!this.timerPanel || !this.remainingTimeElement || !this.elapsedTimeElement) {
            return;
        }

        if (!this.stats || Object.keys(this.stats).length === 0) {
            this.timerPanel.classList.add('hidden');
            if (this.stageTextElement) {
                this.stageTextElement.classList.add('hidden');
            }
            this.remainingTimeElement.textContent = '00:00';
            this.elapsedTimeElement.textContent = '已运行 00:00 / 00:00';
            return;
        }

        const isRunning = this.isRunning && (this.stats.isRunning ?? true);
        const deltaSeconds = isRunning && this.lastStatsUpdate
            ? (Date.now() - this.lastStatsUpdate) / 1000
            : 0;

        const elapsedBase = this.stats.elapsedSeconds ?? 0;
        const totalSeconds = this.stats.totalDurationSeconds ??
            (this.config.duration ? this.config.duration * 60 : 0);
        const remainingBase = this.stats.remainingSeconds ??
            Math.max(0, totalSeconds - elapsedBase);

        const elapsedSeconds = Math.max(0, elapsedBase + deltaSeconds);
        const remainingSeconds = Math.max(0, isRunning ? (remainingBase - deltaSeconds) : remainingBase);

        this.timerPanel.classList.remove('hidden');
        this.remainingTimeElement.textContent = this.formatSeconds(Math.round(remainingSeconds));
        const totalText = totalSeconds > 0 ? this.formatSeconds(Math.round(totalSeconds)) : '--:--';
        this.elapsedTimeElement.textContent = `已运行 ${this.formatSeconds(Math.round(elapsedSeconds))} / ${totalText}`;

        if (this.stageTextElement) {
            const stageLabel = this.getStageDescription(this.stats.stage, this.isRunning);
            if (stageLabel) {
                let stageText = `阶段：${stageLabel}`;
                if (this.stats.lastAction?.message) {
                    stageText += `（最近：${this.stats.lastAction.message}）`;
                }
                this.stageTextElement.textContent = stageText;
                this.stageTextElement.classList.remove('hidden');
            } else {
                this.stageTextElement.classList.add('hidden');
            }
        }
    }

    getStageDescription(stage, running) {
        switch (stage) {
            case 'start':
                return '启动中';
            case 'progress':
                return running ? '执行任务中' : '任务等待';
            case 'resume':
                return '恢复运行';
            case 'completed':
                return '已完成';
            case 'stopping':
                return '正在停止';
            case 'stopped':
                return '已停止';
            default:
                return running ? '正在运行' : '';
        }
    }

    formatSeconds(seconds) {
        if (!Number.isFinite(seconds) || seconds <= 0) {
            return '00:00';
        }

        const hrs = Math.floor(seconds / 3600);
        const mins = Math.floor((seconds % 3600) / 60);
        const secs = Math.floor(seconds % 60);

        if (hrs > 0) {
            return `${String(hrs).padStart(2, '0')}:${String(mins).padStart(2, '0')}:${String(secs).padStart(2, '0')}`;
        }

        return `${String(mins).padStart(2, '0')}:${String(secs).padStart(2, '0')}`;
    }

    formatActionTime(timestamp) {
        if (!timestamp) return '--:--:--';
        const date = new Date(timestamp);
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        const seconds = String(date.getSeconds()).padStart(2, '0');
        return `${hours}:${minutes}:${seconds}`;
    }

    formatActionMessage(action) {
        if (!action) {
            return '';
        }
        return action.message || action.type || '';
    }

    formatActionType(type) {
        if (!type) return '';
        const map = {
            system: '系统',
            scroll: '浏览',
            like: '点赞',
            follow: '关注',
            retweet: '转发',
            error: '错误'
        };
        return map[type] || type;
    }

    renderActionLog(actions = []) {
        if (!this.actionListElement) {
            return;
        }

        this.actionListElement.innerHTML = '';

        if (!actions.length) {
            if (this.emptyLogElement) {
                this.emptyLogElement.classList.remove('hidden');
                this.actionListElement.appendChild(this.emptyLogElement);
            }
            if (this.logSection) {
                this.logSection.style.display = this.isRunning ? 'block' : 'none';
            }
            return;
        }

        if (this.emptyLogElement) {
            this.emptyLogElement.classList.add('hidden');
        }

        const fragment = document.createDocumentFragment();

        actions.forEach(action => {
            const item = document.createElement('div');
            item.className = `action-item ${(action.status || 'info')}`;

            const timeSpan = document.createElement('span');
            timeSpan.className = 'action-time';
            timeSpan.textContent = this.formatActionTime(action.timestamp);

            const messageSpan = document.createElement('span');
            messageSpan.className = 'action-message';
            const typeLabel = this.formatActionType(action.type);
            messageSpan.textContent = typeLabel ? `[${typeLabel}] ${this.formatActionMessage(action)}` : this.formatActionMessage(action);

            item.appendChild(timeSpan);
            item.appendChild(messageSpan);

            const metaParts = [];
            if (typeof action.scrollDistance === 'number') {
                metaParts.push(`滚动 ${action.scrollDistance}px`);
            }
            if (typeof action.durationSeconds !== 'undefined') {
                metaParts.push(`耗时 ${action.durationSeconds}s`);
            }
            if (action.stage) {
                metaParts.push(`阶段 ${this.getStageDescription(action.stage, true) || action.stage}`);
            }
            if (action.configuration) {
                const cfg = action.configuration;
                metaParts.push(
                    `配置: 推文${cfg.tweetsToView ?? '-'} | 点赞${cfg.likesToGive ?? '-'} | 关注${cfg.followsToGive ?? '-'} | 转发${cfg.retweetsToGive ?? '-'} | 时长${cfg.durationMinutes ?? '-'}min`
                );
            }
            if (action.totals) {
                const totals = action.totals;
                metaParts.push(
                    `统计: 推文${totals.tweetsViewed ?? 0} / 点赞${totals.likesGiven ?? 0} / 关注${totals.followsGiven ?? 0} / 转发${totals.retweetsGiven ?? 0}`
                );
            }
            if (action.progress) {
                const progress = action.progress;
                metaParts.push(
                    `完成度: 推${progress.tweetProgress ?? 0}% | 点${progress.likeProgress ?? 0}% | 关${progress.followProgress ?? 0}% | 转${progress.retweetProgress ?? 0}%`
                );
            }

            if (metaParts.length > 0) {
                const meta = document.createElement('div');
                meta.className = 'action-meta';
                meta.textContent = metaParts.join(' · ');
                item.appendChild(meta);
            }

            fragment.appendChild(item);
        });

        this.actionListElement.appendChild(fragment);
        this.actionListElement.scrollTop = this.actionListElement.scrollHeight;
        if (this.logSection) {
            this.logSection.style.display = 'block';
        }
    }

    startStatusUpdate() {
        // 每2秒更新一次状态
        let checking = false;
        setInterval(async () => {
            if (checking) return;
            checking = true;
            try {
                await this.checkStatus();
            } finally {
                checking = false;
            }
        }, 2000);
    }

    showMessage(message, type = 'info') {
        // 创建消息元素
        const messageElement = document.createElement('div');
        messageElement.textContent = message;
        messageElement.style.cssText = `
            position: fixed;
            top: 10px;
            left: 50%;
            transform: translateX(-50%);
            padding: 10px 20px;
            border-radius: 5px;
            color: white;
            font-weight: bold;
            z-index: 1000;
            transition: opacity 0.3s;
        `;

        // 设置颜色
        if (type === 'success') {
            messageElement.style.background = '#28a745';
        } else if (type === 'error') {
            messageElement.style.background = '#dc3545';
        } else {
            messageElement.style.background = '#17a2b8';
        }

        document.body.appendChild(messageElement);

        // 3秒后移除
        setTimeout(() => {
            messageElement.style.opacity = '0';
            setTimeout(() => {
                if (messageElement.parentNode) {
                    messageElement.parentNode.removeChild(messageElement);
                }
            }, 300);
        }, 3000);
    }
}

// 页面加载完成后初始化
document.addEventListener('DOMContentLoaded', () => {
    new PopupController();
});
