// 持久化推特关注机器人 - 解决页面跳转后脚本停止问题
class PersistentFollowBot {
    constructor() {
        this.isRunning = false;
        this.stats = {
            tweetsViewed: 0,
            followsGiven: 0,
            startTime: Date.now()
        };
        this.targets = {
            tweetsToView: 30 + Math.floor(Math.random() * 21), // 30-50条
            followsToGive: 3 + Math.floor(Math.random() * 3)   // 3-5个
        };
        this.currentMethod = null;
        this.retryCount = 0;
        this.maxRetries = 3;
    }

    // 延迟函数
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    // 带倒计时的延迟
    async delayWithCountdown(ms, message) {
        const startTime = Date.now();
        const endTime = startTime + ms;
        
        while (Date.now() < endTime) {
            const remaining = Math.ceil((endTime - Date.now()) / 1000);
            if (remaining > 0) {
                console.log(`⏳ ${message} ${remaining}秒...`);
            }
            await this.delay(1000);
        }
        console.log(`✅ ${message} 完成`);
    }

    // 模拟人类点击
    async simulateHumanClick(element) {
        const rect = element.getBoundingClientRect();
        const x = rect.left + rect.width / 2;
        const y = rect.top + rect.height / 2;
        
        const clickEvent = new MouseEvent('click', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(clickEvent);
    }

    // 检查是否在推特页面
    isOnTwitter() {
        const currentUrl = window.location.href;
        return currentUrl.includes('twitter.com') || currentUrl.includes('x.com');
    }

    // 检查是否在用户主页
    isOnUserProfile() {
        const currentUrl = window.location.href;
        return currentUrl.includes('/') && !currentUrl.includes('/home') && !currentUrl.includes('/search') && !currentUrl.includes('/notifications');
    }

    // 检查是否在搜索结果页
    isOnSearchResults() {
        const currentUrl = window.location.href;
        return currentUrl.includes('/search');
    }

    // 返回推特主页
    async returnToHome() {
        try {
            console.log('🔙 返回推特主页...');
            window.location.href = 'https://twitter.com/home';
            await this.delayWithCountdown(3000, '等待主页加载');
            return true;
        } catch (error) {
            console.error('返回主页失败:', error);
            return false;
        }
    }

    // 在搜索结果页直接关注用户
    async followUserInSearchResults() {
        try {
            console.log('🔍 在搜索结果页查找用户...');
            
            // 查找用户卡片
            const userCards = document.querySelectorAll('[data-testid="UserCell"]');
            console.log(`🔍 找到 ${userCards.length} 个用户卡片`);
            
            if (userCards.length > 0) {
                const userCard = userCards[0];
                console.log('🎯 选择第一个用户卡片');
                
                // 查找关注按钮
                const followButton = userCard.querySelector('button[data-testid="follow"], div[role="button"][data-testid="follow"]');
                
                if (followButton) {
                    console.log('✅ 在搜索结果页找到关注按钮');
                    await this.simulateHumanClick(followButton);
                    console.log('✅ 已点击关注按钮');
                    this.stats.followsGiven++;
                    return true;
                } else {
                    console.log('❌ 在搜索结果页未找到关注按钮，点击用户卡片进入主页');
                    await this.simulateHumanClick(userCard);
                    await this.delayWithCountdown(2000, '等待用户主页加载');
                    return await this.followUserInProfile();
                }
            }
            
            return false;
        } catch (error) {
            console.error('在搜索结果页关注失败:', error);
            return false;
        }
    }

    // 在用户主页关注用户
    async followUserInProfile() {
        try {
            console.log('🔍 在用户主页查找关注按钮...');
            
            // 多种关注按钮选择器
            const followSelectors = [
                'button[data-testid="follow"]',
                'div[role="button"][data-testid="follow"]',
                'button:contains("关注")',
                'div[role="button"]:contains("关注")',
                'button:contains("Follow")',
                'div[role="button"]:contains("Follow")',
                'button[aria-label*="关注"]',
                'div[role="button"][aria-label*="关注"]',
                'button[aria-label*="Follow"]',
                'div[role="button"][aria-label*="Follow"]'
            ];
            
            for (const selector of followSelectors) {
                try {
                    const buttons = document.querySelectorAll(selector);
                    console.log(`🔍 选择器 "${selector}" 找到 ${buttons.length} 个按钮`);
                    
                    for (const button of buttons) {
                        if (button.offsetParent !== null) {
                            const text = button.textContent?.trim() || '';
                            const ariaLabel = button.getAttribute('aria-label') || '';
                            
                            console.log(`🔍 检查按钮: text="${text}", aria-label="${ariaLabel}"`);
                            
                            if (text.includes('关注') || text.includes('Follow') || 
                                ariaLabel.includes('关注') || ariaLabel.includes('Follow')) {
                                console.log(`✅ 找到关注按钮: ${selector}`);
                                await this.simulateHumanClick(button);
                                console.log('✅ 已点击关注按钮');
                                this.stats.followsGiven++;
                                return true;
                            }
                        }
                    }
                } catch (e) {
                    console.log(`❌ 选择器 "${selector}" 出错:`, e.message);
                    continue;
                }
            }
            
            // 广泛搜索所有按钮
            console.log('🔍 使用广泛搜索策略...');
            const allButtons = document.querySelectorAll('button, div[role="button"]');
            console.log(`🔍 页面总共有 ${allButtons.length} 个按钮`);
            
            for (const button of allButtons) {
                const text = button.textContent?.trim() || '';
                const ariaLabel = button.getAttribute('aria-label') || '';
                
                if ((text === '关注' || text === 'Follow' || 
                     ariaLabel.includes('关注') || ariaLabel.includes('Follow')) &&
                    button.offsetParent !== null) {
                    console.log(`✅ 广泛搜索找到关注按钮: text="${text}", aria-label="${ariaLabel}"`);
                    await this.simulateHumanClick(button);
                    console.log('✅ 已点击关注按钮');
                    this.stats.followsGiven++;
                    return true;
                }
            }
            
            return false;
        } catch (error) {
            console.error('在用户主页关注失败:', error);
            return false;
        }
    }

    // 搜索用户
    async searchUser(username) {
        try {
            console.log(`🔍 搜索用户: @${username}`);
            
            // 导航到搜索页面
            const searchUrl = `https://x.com/search?q=${encodeURIComponent(username)}&src=typed_query&f=user`;
            window.location.href = searchUrl;
            
            await this.delayWithCountdown(3000, '等待搜索结果加载');
            return true;
        } catch (error) {
            console.error('搜索用户失败:', error);
            return false;
        }
    }

    // 自然浏览
    async scrollAndBrowse() {
        try {
            const scrollDistance = Math.random() * 300 + 300;
            
            window.scrollBy({
                top: scrollDistance,
                behavior: 'smooth'
            });
            
            console.log(`📜 向下滑动 ${Math.round(scrollDistance)}px，浏览内容...`);
            this.stats.tweetsViewed++;
            
            const isInterested = Math.random() < 0.3;
            
            if (isInterested) {
                const readingTime = Math.random() * 4 + 2;
                console.log(`👀 发现感兴趣的内容！阅读时间: ${readingTime.toFixed(1)}秒`);
                await this.delayWithCountdown(readingTime * 1000, '阅读感兴趣的内容');
            } else {
                const skipTime = Math.random() * 1.5 + 0.5;
                console.log(`⏭️ 内容不感兴趣，快速跳过 (${skipTime.toFixed(1)}秒)`);
                await this.delayWithCountdown(skipTime * 1000, '快速跳过不感兴趣内容');
            }
            
            return true;
        } catch (error) {
            console.error('自然浏览失败:', error);
            return false;
        }
    }

    // 显示进度
    showProgress() {
        const elapsed = Math.round((Date.now() - this.stats.startTime) / 1000);
        console.log(`📊 进度: ${elapsed}秒 | 推文:${this.stats.tweetsViewed}/${this.targets.tweetsToView} | 关注:${this.stats.followsGiven}/${this.targets.followsToGive}`);
    }

    // 页面状态检测和自动恢复
    async checkPageStateAndRecover() {
        if (!this.isOnTwitter()) {
            console.log('⚠️ 不在推特页面，尝试返回...');
            return await this.returnToHome();
        }
        
        if (this.isOnSearchResults()) {
            console.log('🔍 检测到搜索结果页，尝试直接关注...');
            const success = await this.followUserInSearchResults();
            if (success) {
                console.log('✅ 在搜索结果页关注成功！');
                await this.delayWithCountdown(2000, '关注完成，等待继续');
                return await this.returnToHome();
            } else {
                console.log('❌ 在搜索结果页关注失败，返回主页');
                return await this.returnToHome();
            }
        }
        
        if (this.isOnUserProfile()) {
            console.log('👤 检测到用户主页，尝试关注...');
            const success = await this.followUserInProfile();
            if (success) {
                console.log('✅ 在用户主页关注成功！');
                await this.delayWithCountdown(2000, '关注完成，等待继续');
                return await this.returnToHome();
            } else {
                console.log('❌ 在用户主页关注失败，返回主页');
                return await this.returnToHome();
            }
        }
        
        return true;
    }

    // 开始关注流程
    async startFollowProcess() {
        if (this.isRunning) {
            console.log('⚠️ 关注流程已在进行中');
            return;
        }
        
        this.isRunning = true;
        console.log('🚀 开始持久化关注流程...');
        console.log(`🎯 目标: 浏览${this.targets.tweetsToView}条推文, 关注${this.targets.followsToGive}个用户`);
        
        const startTime = Date.now();
        
        while (this.stats.followsGiven < this.targets.followsToGive && this.isRunning) {
            try {
                // 检查页面状态并自动恢复
                const pageStateOk = await this.checkPageStateAndRecover();
                if (!pageStateOk) {
                    console.log('❌ 页面状态检查失败，等待重试...');
                    await this.delay(3000);
                    continue;
                }
                
                // 随机选择关注方法
                const methods = [
                    () => this.searchUser('liuruyy'),
                    () => this.searchUser('randomuser'),
                    () => this.searchUser('testuser'),
                    () => this.scrollAndBrowse()
                ];
                
                const randomMethod = methods[Math.floor(Math.random() * methods.length)];
                this.currentMethod = randomMethod.name || 'unknown';
                
                console.log(`🔄 尝试关注用户 (方法: ${this.currentMethod})...`);
                const success = await randomMethod();
                
                if (success) {
                    console.log('✅ 操作成功！');
                    await this.delayWithCountdown(3000, '操作完成，等待继续');
                } else {
                    console.log('❌ 操作失败，尝试浏览内容...');
                    await this.scrollAndBrowse();
                    await this.delayWithCountdown(2000, '浏览完成，等待继续');
                }
                
                this.showProgress();
                
            } catch (error) {
                console.error('关注流程出错:', error);
                this.retryCount++;
                
                if (this.retryCount >= this.maxRetries) {
                    console.log('❌ 达到最大重试次数，停止脚本');
                    break;
                }
                
                console.log(`🔄 重试 ${this.retryCount}/${this.maxRetries}...`);
                await this.delay(5000);
            }
        }
        
        console.log('✅ 关注流程完成!');
        console.log(`📈 最终统计: 浏览${this.stats.tweetsViewed}条推文, 关注${this.stats.followsGiven}个用户`);
        this.isRunning = false;
    }

    // 停止关注流程
    stopFollowProcess() {
        this.isRunning = false;
        console.log('⏹️ 关注流程已停止');
    }
}

// 创建实例
const persistentBot = new PersistentFollowBot();

// 全局控制
window.persistentFollowBot = {
    start: () => persistentBot.startFollowProcess(),
    stop: () => persistentBot.stopFollowProcess(),
    status: () => console.log('📊 状态:', {
        isRunning: persistentBot.isRunning,
        stats: persistentBot.stats,
        currentMethod: persistentBot.currentMethod
    })
};

// 页面跳转检测和自动恢复
function handlePageNavigation() {
    console.log('🔄 检测到页面变化，检查状态...');
    
    if (persistentBot.isRunning) {
        console.log('🤖 脚本正在运行，自动恢复...');
        setTimeout(() => {
            persistentBot.checkPageStateAndRecover();
        }, 2000);
    }
}

// 监听页面变化
window.addEventListener('popstate', handlePageNavigation);
window.addEventListener('pushstate', handlePageNavigation);
window.addEventListener('replacestate', handlePageNavigation);

// 监听URL变化
let currentUrl = window.location.href;
setInterval(() => {
    if (window.location.href !== currentUrl) {
        currentUrl = window.location.href;
        console.log('🔄 检测到URL变化:', currentUrl);
        handlePageNavigation();
    }
}, 1000);

// 页面加载完成后检查
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', handlePageNavigation);
} else {
    handlePageNavigation();
}

console.log('🤖 持久化关注机器人已加载');
console.log('📋 使用方法:');
console.log('   persistentFollowBot.start() - 开始关注流程');
console.log('   persistentFollowBot.stop() - 停止关注流程');
console.log('   persistentFollowBot.status() - 查看状态');

// 自动开始
persistentBot.startFollowProcess();
