#!/usr/bin/env python3

"""
Generate Chrome extension icons by stylising the provided screenshot with an X-inspired motif.

Usage:
    python generate_x_style_icons.py
"""

from __future__ import annotations

from pathlib import Path

from PIL import Image, ImageDraw, ImageEnhance, ImageFilter, ImageOps

# Absolute path to the source screenshot provided by the user.
SOURCE_PATH = Path(r"C:\Users\10902\Pictures\企业微信截图_20251024174423.png")

# Where the generated icons will be written.
OUTPUT_DIR = Path(__file__).resolve().parent / "icons"

# Sizes required by the Chrome extension.
ICON_SIZES = (16, 32, 48, 128, 256, 512)


def _build_background(image: Image.Image, size: int) -> Image.Image:
    """Create a moody, X-style background using the screenshot's content."""
    base = ImageOps.fit(image, (size, size), method=Image.Resampling.LANCZOS)
    softened = base.filter(ImageFilter.GaussianBlur(radius=size * 0.02))
    toned = ImageEnhance.Color(softened).enhance(0.35)
    toned = ImageEnhance.Contrast(toned).enhance(0.9)
    toned = ImageEnhance.Brightness(toned).enhance(0.55)

    # Blend with a charcoal overlay to lean into X's dark aesthetic.
    charcoal = Image.new("RGBA", (size, size), (8, 10, 12, 255))
    background = Image.blend(toned, charcoal, alpha=0.75)

    # Restore a soft focus of the original content in the centre.
    spotlight = Image.new("L", (size, size), 0)
    draw = ImageDraw.Draw(spotlight)
    inset = size * 0.16
    draw.ellipse((inset, inset, size - inset, size - inset), fill=230)
    spotlight = spotlight.filter(ImageFilter.GaussianBlur(radius=size * 0.14))
    subtle_foreground = ImageEnhance.Brightness(base).enhance(1.15)
    subtle_foreground = ImageEnhance.Color(subtle_foreground).enhance(0.45)

    return Image.composite(subtle_foreground, background, spotlight)


def _draw_x_stroke(draw: ImageDraw.ImageDraw, size: int, slope: int, width: float, color: tuple[int, int, int, int]) -> None:
    """Render a single diagonal stroke used to assemble the X logomark."""
    offset = width / 2

    # Pre-compute endpoints on the diagonal before trimming to canvas bounds.
    if slope > 0:
        start = (-offset, size + offset)
        end = (size + offset, -offset)
    else:
        start = (-offset, -offset)
        end = (size + offset, size + offset)

    draw.line([start, end], fill=color, width=int(width))


def _build_x_overlay(size: int) -> Image.Image:
    """Create the layered X mark that will sit atop the background."""
    canvas = Image.new("RGBA", (size, size), (0, 0, 0, 0))
    draw = ImageDraw.Draw(canvas)

    primary_width = size * 0.24
    accent_width = size * 0.14

    _draw_x_stroke(draw, size, slope=1, width=primary_width, color=(250, 250, 250, 245))
    _draw_x_stroke(draw, size, slope=-1, width=primary_width, color=(250, 250, 250, 245))

    # Cool accent strokes to introduce depth.
    accent = Image.new("RGBA", (size, size), (0, 0, 0, 0))
    accent_draw = ImageDraw.Draw(accent)
    accent_color = (140, 185, 255, 200)
    accent_draw.line(
        [(size * 0.08, size * 0.88), (size * 0.82, size * 0.18)],
        fill=accent_color,
        width=int(accent_width),
    )
    accent_draw.line(
        [(size * 0.18, size * 0.08), (size * 0.92, size * 0.82)],
        fill=accent_color,
        width=int(accent_width * 0.85),
    )
    accent = accent.filter(ImageFilter.GaussianBlur(radius=size * 0.01))

    combined = Image.alpha_composite(canvas, accent)
    softened = combined.filter(ImageFilter.GaussianBlur(radius=size * 0.006))
    return softened


def build_master_icon(source: Image.Image, size: int = 512) -> Image.Image:
    """Combine the processed background and X motif into a master icon."""
    background = _build_background(source, size)
    overlay = _build_x_overlay(size)
    master = Image.alpha_composite(background, overlay)

    # Add a faint inner glow to keep the icon legible on dark toolbars.
    glow = Image.new("RGBA", (size, size), (255, 255, 255, 0))
    draw = ImageDraw.Draw(glow)
    inset = size * 0.04
    draw.rounded_rectangle(
        (inset, inset, size - inset, size - inset),
        radius=size * 0.12,
        outline=(255, 255, 255, 96),
        width=int(size * 0.01),
    )
    glow = glow.filter(ImageFilter.GaussianBlur(radius=size * 0.02))

    return Image.alpha_composite(master, glow)


def main() -> None:
    if not SOURCE_PATH.exists():
        raise FileNotFoundError(f"Source image not found: {SOURCE_PATH}")

    OUTPUT_DIR.mkdir(exist_ok=True)
    source_image = Image.open(SOURCE_PATH).convert("RGBA")

    master_size = max(ICON_SIZES)
    master_icon = build_master_icon(source_image, size=master_size)

    for size in ICON_SIZES:
        resized = master_icon.resize((size, size), Image.Resampling.LANCZOS)
        target = OUTPUT_DIR / f"icon{size}.png"
        resized.save(target, format="PNG")
        print(f"Saved {target} ({size}x{size})")


if __name__ == "__main__":
    main()
