// 推特养号助手 - 修复版（解决页面跳转后脚本停止问题）
class TwitterWarmupBot {
    constructor() {
        this.isRunning = false;
        this.stats = {
            tweetsViewed: 0,
            likesGiven: 0,
            followsGiven: 0,
            retweetsGiven: 0,
            hoverCardsTriggered: 0,
            startTime: Date.now(),
            actionSequence: []
        };
        this.targets = {
            tweetsToView: 60 + Math.floor(Math.random() * 41), // 60-100条
            likesToGive: 0 + Math.floor(Math.random() * 0),     // 0条
            followsToGive: 3 + Math.floor(Math.random() * 3),   // 3-5个
            retweetsToGive: 0 + Math.floor(Math.random() * 0)   // 0条转发
        };
        this.duration = (20 + Math.random() * 10) * 60 * 1000; // 20-30分钟
        this.currentAction = null;
        this.pageNavigationHandler = null;
    }

    // 延迟函数
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    // 带倒计时的延迟
    async delayWithCountdown(ms, message) {
        const startTime = Date.now();
        const endTime = startTime + ms;
        
        while (Date.now() < endTime) {
            const remaining = Math.ceil((endTime - Date.now()) / 1000);
            if (remaining > 0) {
                console.log(`⏳ ${message} ${remaining}秒...`);
            }
            await this.delay(1000);
        }
        console.log(`✅ ${message} 完成`);
    }

    // 模拟人类点击
    async simulateHumanClick(element) {
        const rect = element.getBoundingClientRect();
        const x = rect.left + rect.width / 2;
        const y = rect.top + rect.height / 2;
        
        const clickEvent = new MouseEvent('click', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(clickEvent);
    }

    // 检查是否在推特页面
    checkPageNavigation() {
        const currentUrl = window.location.href;
        const isTwitterHome = currentUrl.includes('twitter.com') || currentUrl.includes('x.com');
        
        if (!isTwitterHome) {
            console.log('⚠️ 检测到页面跳转，当前URL:', currentUrl);
            return false;
        }
        return true;
    }

    // 安全返回推特主页
    async safeReturnToTwitter() {
        try {
            console.log('🔙 尝试返回推特主页...');
            
            if (window.history.length > 1) {
                window.history.back();
                await this.delay(2000);
                
                if (this.checkPageNavigation()) {
                    console.log('✅ 成功返回推特主页');
                    return true;
                }
            }
            
            console.log('🔙 返回失败，直接导航到推特主页...');
            window.location.href = 'https://twitter.com/home';
            await this.delay(3000);
            
            return this.checkPageNavigation();
        } catch (error) {
            console.error('返回推特主页失败:', error);
            return false;
        }
    }

    // 自然浏览行为
    async scrollAndBrowse() {
        try {
            const scrollDistance = Math.random() * 300 + 300;
            
            window.scrollBy({
                top: scrollDistance,
                behavior: 'smooth'
            });
            
            console.log(`📜 向下滑动 ${Math.round(scrollDistance)}px，浏览内容...`);
            this.stats.tweetsViewed++;
            
            const isInterested = Math.random() < 0.3;
            
            if (isInterested) {
                const readingTime = Math.random() * 4 + 2;
                console.log(`👀 发现感兴趣的内容！阅读时间: ${readingTime.toFixed(1)}秒`);
                await this.delayWithCountdown(readingTime * 1000, '阅读感兴趣的内容');
                
                const continueDelay = Math.random() * 1000 + 500;
                await this.delayWithCountdown(continueDelay, '阅读完成，等待继续浏览');
            } else {
                const skipTime = Math.random() * 1.5 + 0.5;
                console.log(`⏭️ 内容不感兴趣，快速跳过 (${skipTime.toFixed(1)}秒)`);
                await this.delayWithCountdown(skipTime * 1000, '快速跳过不感兴趣内容');
                
                const skipDelay = Math.random() * 500 + 200;
                await this.delayWithCountdown(skipDelay, '跳过完成，等待继续浏览');
            }
            
            if (Math.random() < 0.08) {
                const backScroll = 50 + Math.random() * 100;
                window.scrollBy({
                    top: -backScroll,
                    behavior: 'smooth'
                });
                console.log(`↩️ 回看内容 ${Math.round(backScroll)}px`);
                
                const backScrollDelay = 1000 + Math.random() * 2000;
                await this.delayWithCountdown(backScrollDelay, '回看完成，等待继续');
            }
            
            return true;
        } catch (error) {
            console.error('自然浏览失败:', error);
            return false;
        }
    }

    // 点击头像进入主页关注功能
    async hoverCardFollow() {
        try {
            console.log('🔍 开始点击头像进入主页关注流程...');
            
            // 头像选择器
            const avatarSelectors = [
                '[data-testid="UserAvatar-Container"]',
                '[data-testid="UserAvatar"]',
                '[data-testid="primaryColumn"] [data-testid="UserAvatar-Container"]',
                '[data-testid="primaryColumn"] [data-testid="UserAvatar"]',
                'article [data-testid="UserAvatar-Container"]',
                'article [data-testid="UserAvatar"]',
                'div[role="button"][aria-label*="头像"]',
                'div[role="button"][aria-label*="Avatar"]',
                'div[role="button"][aria-label*="Profile"]',
                'div[class*="css-175oi2r"][class*="r-172uzmj"]',
                'div[class*="UserAvatar"]',
                'div[role="button"][class*="avatar"]',
                'div[class*="css-175oi2r"] img[alt*="头像"]',
                'div[class*="css-175oi2r"] img[alt*="Avatar"]',
                'div[class*="css-175oi2r"] img[alt*="Profile"]'
            ];
            
            let avatarElements = [];
            for (const selector of avatarSelectors) {
                const elements = document.querySelectorAll(selector);
                avatarElements = avatarElements.concat(Array.from(elements));
            }
            
            // 过滤可见的头像
            const basicAvatars = avatarElements.filter(avatar => {
                const isVisible = avatar.offsetParent !== null;
                const hasSize = avatar.offsetWidth > 0 && avatar.offsetHeight > 0;
                return isVisible && hasSize;
            });
            
            console.log(`📊 基础过滤后找到 ${basicAvatars.length} 个头像`);
            
            const visibleAvatars = basicAvatars.filter(avatar => {
                const isNotSidebar = !avatar.closest('[data-testid="sidebarColumn"]') && 
                                   !avatar.closest('[role="navigation"]') &&
                                   !avatar.closest('nav') &&
                                   !avatar.closest('[data-testid="SideNav"]');
                
                const isInTweetArea = avatar.closest('[data-testid="primaryColumn"]') ||
                                     avatar.closest('[data-testid="tweet"]') ||
                                     avatar.closest('article') ||
                                     avatar.closest('[role="article"]');
                
                return isNotSidebar && isInTweetArea;
            });
            
            console.log(`📊 找到 ${visibleAvatars.length} 个可见可点击头像`);
            
            if (visibleAvatars.length === 0) {
                console.log('❌ 未找到任何头像元素，尝试备用方法...');
                
                const fallbackSelectors = [
                    'img[alt*="头像"]',
                    'img[alt*="Avatar"]',
                    'img[alt*="Profile"]',
                    'div[role="button"] img',
                    'article img[alt]',
                    '[data-testid="primaryColumn"] img[alt]'
                ];
                
                let fallbackAvatars = [];
                for (const selector of fallbackSelectors) {
                    const elements = document.querySelectorAll(selector);
                    fallbackAvatars = fallbackAvatars.concat(Array.from(elements));
                }
                
                const visibleFallbackAvatars = fallbackAvatars.filter(avatar => {
                    const isVisible = avatar.offsetParent !== null;
                    const hasSize = avatar.offsetWidth > 0 && avatar.offsetHeight > 0;
                    const isNotSidebar = !avatar.closest('[data-testid="sidebarColumn"]') && 
                                       !avatar.closest('[role="navigation"]') &&
                                       !avatar.closest('nav') &&
                                       !avatar.closest('[data-testid="SideNav"]');
                    const isInTweetArea = avatar.closest('[data-testid="primaryColumn"]') ||
                                       avatar.closest('[data-testid="tweet"]') ||
                                       avatar.closest('article') ||
                                       avatar.closest('[role="article"]');
                    
                    return isVisible && hasSize && isNotSidebar && isInTweetArea;
                });
                
                console.log(`📊 备用方法找到 ${visibleFallbackAvatars.length} 个头像`);
                
                if (visibleFallbackAvatars.length === 0) {
                    console.log('❌ 备用方法也未找到头像元素');
                    return false;
                }
                
                const randomAvatar = visibleFallbackAvatars[Math.floor(Math.random() * visibleFallbackAvatars.length)];
                console.log('🎯 使用备用方法选择的头像:', {
                    tagName: randomAvatar.tagName,
                    alt: randomAvatar.getAttribute('alt'),
                    className: randomAvatar.className
                });
                
                await this.simulateHumanClick(randomAvatar);
                console.log('✅ 已点击头像，进入用户主页');
                
                await this.delayWithCountdown(2000, '等待用户主页加载');
                
                return await this.continueFollowProcess();
            }
            
            const randomAvatar = visibleAvatars[Math.floor(Math.random() * visibleAvatars.length)];
            console.log('🎯 选择的头像元素:', {
                className: randomAvatar.className,
                role: randomAvatar.getAttribute('role'),
                ariaLabel: randomAvatar.getAttribute('aria-label'),
                tagName: randomAvatar.tagName
            });
            
            randomAvatar.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(1000);
            
            const beforeUrl = window.location.href;
            console.log('📍 点击前URL:', beforeUrl);
            
            await this.simulateHumanClick(randomAvatar);
            console.log('✅ 已点击头像，等待页面跳转...');
            
            await this.delayWithCountdown(3000, '等待页面跳转');
            
            const afterUrl = window.location.href;
            console.log('📍 点击后URL:', afterUrl);
            
            if (beforeUrl === afterUrl) {
                console.log('⚠️ 页面未跳转，可能点击失败');
                return false;
            }
            
            console.log('✅ 成功进入用户主页');
            
            await this.delayWithCountdown(2000, '等待页面完全加载');
            
            if (document.readyState !== 'complete') {
                console.log('⏳ 页面仍在加载，等待完成...');
                await this.delayWithCountdown(3000, '等待页面加载完成');
            }
            
            console.log('🔄 开始关注流程...');
            return await this.continueFollowProcess();
            
        } catch (error) {
            console.error('点击头像关注失败:', error);
            
            try {
                console.log('🔄 尝试返回原页面...');
                window.history.back();
                await this.delay(2000);
                console.log('✅ 已返回原页面');
            } catch (e) {
                console.log('❌ 返回失败:', e);
            }
            
            return false;
        }
    }

    // 继续关注流程
    async continueFollowProcess() {
        try {
            const currentUrl = window.location.href;
            console.log('📍 当前页面URL:', currentUrl);
            
            console.log('📜 在用户主页浏览内容...');
            
            const browseCount = 2 + Math.floor(Math.random() * 3);
            
            for (let i = 0; i < browseCount; i++) {
                const scrollDistance = Math.random() * 400 + 200;
                window.scrollBy({
                    top: scrollDistance,
                    behavior: 'smooth'
                });
                console.log(`📜 主页浏览滚动 ${Math.round(scrollDistance)}px (${i + 1}/${browseCount})`);
                
                const readingTime = Math.random() * 2 + 1;
                await this.delayWithCountdown(readingTime * 1000, '浏览用户内容');
            }
            
            console.log('🔍 在用户主页查找关注按钮...');
            
            const followSelectors = [
                'button[data-testid="follow"]',
                'div[role="button"][data-testid="follow"]',
                'button:contains("关注")',
                'div[role="button"]:contains("关注")',
                'button:contains("Follow")',
                'div[role="button"]:contains("Follow")',
                'button[aria-label*="关注"]',
                'div[role="button"][aria-label*="关注"]',
                'button[aria-label*="Follow"]',
                'div[role="button"][aria-label*="Follow"]',
                'button[class*="follow"]',
                'div[role="button"][class*="follow"]',
                'button[class*="Follow"]',
                'div[role="button"][class*="Follow"]'
            ];
            
            let followButton = null;
            let foundWithSelector = '';
            
            for (const selector of followSelectors) {
                try {
                    const buttons = document.querySelectorAll(selector);
                    console.log(`🔍 选择器 "${selector}" 找到 ${buttons.length} 个按钮`);
                    
                    for (const button of buttons) {
                        if (button.offsetParent !== null) {
                            const text = button.textContent?.trim() || '';
                            const ariaLabel = button.getAttribute('aria-label') || '';
                            
                            console.log(`🔍 检查按钮: text="${text}", aria-label="${ariaLabel}"`);
                            
                            if (text.includes('关注') || text.includes('Follow') || 
                                ariaLabel.includes('关注') || ariaLabel.includes('Follow')) {
                                followButton = button;
                                foundWithSelector = selector;
                                console.log(`✅ 找到关注按钮，使用选择器: ${selector}`);
                                break;
                            }
                        }
                    }
                    if (followButton) break;
                } catch (e) {
                    console.log(`❌ 选择器 "${selector}" 出错:`, e.message);
                    continue;
                }
            }
            
            if (!followButton) {
                console.log('🔍 使用更广泛的搜索策略...');
                const allButtons = document.querySelectorAll('button, div[role="button"]');
                console.log(`🔍 页面总共有 ${allButtons.length} 个按钮`);
                
                for (const button of allButtons) {
                    const text = button.textContent?.trim() || '';
                    const ariaLabel = button.getAttribute('aria-label') || '';
                    
                    if ((text === '关注' || text === 'Follow' || 
                         ariaLabel.includes('关注') || ariaLabel.includes('Follow')) &&
                        button.offsetParent !== null) {
                        followButton = button;
                        foundWithSelector = '广泛搜索';
                        console.log(`✅ 广泛搜索找到关注按钮: text="${text}", aria-label="${ariaLabel}"`);
                        break;
                    }
                }
            }
            
            if (followButton) {
                console.log('✅ 在用户主页找到关注按钮:', {
                    textContent: followButton.textContent?.trim(),
                    ariaLabel: followButton.getAttribute('aria-label'),
                    className: followButton.className,
                    foundWith: foundWithSelector
                });
                
                followButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
                await this.delay(500);
                
                await this.simulateHumanClick(followButton);
                console.log('✅ 已点击关注按钮');
                
                this.stats.followsGiven++;
                
                await this.delayWithCountdown(2000, '关注操作完成');
                
                console.log('🔙 返回上一页...');
                window.history.back();
                await this.delayWithCountdown(2000, '等待返回完成');
                
                console.log('✅ 关注流程完成，已返回');
                return true;
            } else {
                console.log('❌ 在用户主页未找到关注按钮');
                
                const allButtons = document.querySelectorAll('button, div[role="button"]');
                console.log('🔍 页面上的所有按钮:');
                allButtons.forEach((button, index) => {
                    if (index < 10) {
                        console.log(`  按钮${index + 1}: text="${button.textContent?.trim()}", aria-label="${button.getAttribute('aria-label')}", visible=${button.offsetParent !== null}`);
                    }
                });
                
                console.log('🔙 未找到关注按钮，返回上一页...');
                window.history.back();
                await this.delay(1000);
                
                return false;
            }
            
        } catch (error) {
            console.error('继续关注流程失败:', error);
            
            try {
                window.history.back();
            } catch (e) {
                console.log('返回失败:', e);
            }
            
            return false;
        }
    }

    // 智能计算等待时间
    calculateWaitTime(action) {
        let baseTime = 4000;
        
        switch (action) {
            case 'scroll':
                baseTime = 1500 + Math.random() * 2000;
                break;
            case 'like':
                baseTime = 2000 + Math.random() * 2000;
                break;
            case 'hoverFollow':
            case 'follow':
                baseTime = 4000 + Math.random() * 4000;
                break;
            case 'retweet':
                baseTime = 5000 + Math.random() * 3000;
                break;
        }
        
        const recentActions = this.stats.actionSequence.slice(-2);
        if (recentActions.includes(action)) {
            baseTime *= 1.5;
        }
        
        return Math.round(baseTime);
    }

    // 检查动作间隔
    shouldAvoidAction(action) {
        const recentActions = this.stats.actionSequence.slice(-3);
        const sameActionCount = recentActions.filter(a => a === action).length;
        return sameActionCount >= 2;
    }

    // 检查是否需要强制互动
    shouldForceInteraction() {
        const totalInteractions = this.targets.likesToGive + this.targets.followsToGive + this.targets.retweetsToGive;
        const completedInteractions = this.stats.likesGiven + this.stats.followsGiven + this.stats.retweetsGiven;
        const elapsed = (Date.now() - this.stats.startTime) / 1000 / 60;
        
        if (elapsed > 5 && (completedInteractions / totalInteractions) < 0.2) {
            return true;
        }
        
        const recentActions = this.stats.actionSequence.slice(-10);
        if (recentActions.length >= 10 && recentActions.every(action => action === 'scroll')) {
            return true;
        }
        
        return false;
    }

    // 选择下一个动作
    selectNextAction() {
        const actions = [];
        
        for (let i = 0; i < 15; i++) {
            actions.push('scroll');
        }
        
        const availableActions = [];
        
        if (this.stats.likesGiven < this.targets.likesToGive && 
            !this.shouldAvoidAction('like')) {
            availableActions.push('like');
        }
        
        if (this.stats.followsGiven < this.targets.followsToGive) {
            const followMethods = [];
            if (!this.shouldAvoidAction('hoverFollow')) {
                followMethods.push('hoverFollow');
            }
            if (!this.shouldAvoidAction('follow')) {
                followMethods.push('follow');
            }
            
            if (followMethods.length > 0) {
                const selectedMethod = followMethods[Math.floor(Math.random() * followMethods.length)];
                availableActions.push(selectedMethod);
            }
        }
        
        if (this.stats.retweetsGiven < this.targets.retweetsToGive && 
            !this.shouldAvoidAction('retweet')) {
            availableActions.push('retweet');
        }
        
        const forceInteraction = this.shouldForceInteraction();
        
        if (forceInteraction && availableActions.length > 0) {
            console.log('🚨 强制互动模式激活！');
            availableActions.forEach(action => {
                actions.push(action);
                actions.push(action);
                actions.push(action);
            });
        } else {
            availableActions.forEach(action => {
                let weight = 1;
                
                switch (action) {
                    case 'like':
                        const likeProgress = this.stats.likesGiven / this.targets.likesToGive;
                        if (likeProgress >= 0.8) weight = 0.5;
                        else if (likeProgress >= 0.6) weight = 0.7;
                        break;
                    case 'hoverFollow':
                    case 'follow':
                        const followProgress = this.stats.followsGiven / this.targets.followsToGive;
                        if (followProgress < 0.2) weight = 2;
                        else if (followProgress < 0.5) weight = 1.5;
                        break;
                    case 'retweet':
                        const retweetProgress = this.stats.retweetsGiven / this.targets.retweetsToGive;
                        if (retweetProgress < 0.3) weight = 2;
                        else if (retweetProgress < 0.6) weight = 1.5;
                        break;
                }
                
                const maxInteractions = Math.min(Math.round(weight * 2), 3);
                for (let i = 0; i < maxInteractions; i++) {
                    actions.push(action);
                }
            });
        }
        
        if (this.stats.likesGiven >= this.targets.likesToGive && 
            this.stats.followsGiven >= this.targets.followsToGive && 
            this.stats.retweetsGiven >= this.targets.retweetsToGive) {
            for (let i = 0; i < 8; i++) {
                actions.push('scroll');
            }
        }
        
        const likeProgress = this.stats.likesGiven / this.targets.likesToGive;
        const followProgress = this.stats.followsGiven / this.targets.followsToGive;
        const retweetProgress = this.stats.retweetsGiven / this.targets.retweetsToGive;
        
        let selectedAction;
        
        if (likeProgress >= 0.8 && (followProgress < 0.5 || retweetProgress < 0.5)) {
            const nonLikeActions = actions.filter(action => action !== 'like' && action !== 'scroll');
            if (nonLikeActions.length > 0) {
                selectedAction = nonLikeActions[Math.floor(Math.random() * nonLikeActions.length)];
                console.log(`⚖️ 强制平衡：点赞完成度${Math.round(likeProgress*100)}%，选择其他动作: ${selectedAction}`);
            } else {
                selectedAction = actions[Math.floor(Math.random() * actions.length)];
            }
        } else {
            selectedAction = actions[Math.floor(Math.random() * actions.length)];
        }
        
        if (this.stats.actionSequence.length % 5 === 0) {
            const scrollCount = actions.filter(a => a === 'scroll').length;
            const likeCount = actions.filter(a => a === 'like').length;
            const followCount = actions.filter(a => a === 'hoverFollow' || a === 'follow').length;
            const retweetCount = actions.filter(a => a === 'retweet').length;
            
            const scrollRatio = Math.round((scrollCount / actions.length) * 100);
            
            console.log(`🔍 动作选择调试: 总动作数=${actions.length}, 滚动=${scrollCount}(${scrollRatio}%), 点赞=${likeCount}, 关注=${followCount}, 转发=${retweetCount}`);
            console.log(`🎯 可用互动选项:`, availableActions);
            console.log(`📊 当前统计: 点赞${this.stats.likesGiven}/${this.targets.likesToGive}(${Math.round(this.stats.likesGiven/this.targets.likesToGive*100)}%), 关注${this.stats.followsGiven}/${this.targets.followsToGive}(${Math.round(this.stats.followsGiven/this.targets.followsToGive*100)}%), 转发${this.stats.retweetsGiven}/${this.targets.retweetsToGive}(${Math.round(this.stats.retweetsGiven/this.targets.retweetsToGive*100)}%)`);
            
            if (scrollRatio < 60) {
                console.log('⚠️ 滚动频率偏低，建议增加滚动动作');
            }
        }
        
        this.stats.actionSequence = this.stats.actionSequence || [];
        this.stats.actionSequence.push(selectedAction);
        if (this.stats.actionSequence.length > 10) {
            this.stats.actionSequence.shift();
        }
        
        return selectedAction;
    }

    // 显示进度
    showProgress() {
        const elapsed = Math.round((Date.now() - this.stats.startTime) / 1000);
        const remaining = Math.max(0, Math.round((this.duration - (Date.now() - this.stats.startTime)) / 1000));
        const totalInteractions = this.targets.likesToGive + this.targets.followsToGive + this.targets.retweetsToGive;
        const completedInteractions = this.stats.likesGiven + this.stats.followsGiven + this.stats.retweetsGiven;
        const interactionProgress = Math.round((completedInteractions / totalInteractions) * 100);
        
        const recentActions = this.stats.actionSequence.slice(-3).join(' → ');
        
        console.log(`📊 预热进度: ${elapsed}秒 | 推文:${this.stats.tweetsViewed}/${this.targets.tweetsToView} | 点赞:${this.stats.likesGiven}/${this.targets.likesToGive} | 关注:${this.stats.followsGiven}/${this.targets.followsToGive} | 转发:${this.stats.retweetsGiven}/${this.targets.retweetsToGive} | 悬停:${this.stats.hoverCardsTriggered} | 互动进度:${interactionProgress}% | 最近动作:${recentActions} | 剩余:${remaining}秒`);
    }

    // 显示最终统计
    showFinalStats() {
        const totalTime = Math.round((Date.now() - this.stats.startTime) / 1000);
        
        console.log('✅ 浏览预热完成!');
        console.log(`📈 最终统计:`);
        console.log(`   总时间: ${totalTime}秒 (${Math.round(totalTime/60)}分钟)`);
        console.log(`   浏览推文: ${this.stats.tweetsViewed}条`);
        console.log(`   点赞: ${this.stats.likesGiven}条`);
        console.log(`   关注: ${this.stats.followsGiven}个账号`);
        console.log(`   转发: ${this.stats.retweetsGiven}条`);
        console.log(`   触发悬停卡片: ${this.stats.hoverCardsTriggered}次`);
        
        const tweetProgress = Math.round(this.stats.tweetsViewed/this.targets.tweetsToView*100);
        const likeProgress = Math.round(this.stats.likesGiven/this.targets.likesToGive*100);
        const followProgress = Math.round(this.stats.followsGiven/this.targets.followsToGive*100);
        const retweetProgress = Math.round(this.stats.retweetsGiven/this.targets.retweetsToGive*100);
        
        console.log(`   完成度: 推文${tweetProgress}% | 点赞${likeProgress}% | 关注${followProgress}% | 转发${retweetProgress}%`);
        
        const actionCounts = {};
        this.stats.actionSequence.forEach(action => {
            actionCounts[action] = (actionCounts[action] || 0) + 1;
        });
        console.log(`   动作分布:`, actionCounts);
        
        const actionSequenceStr = this.stats.actionSequence.slice(-10).join(' → ');
        console.log(`   最近动作序列: ${actionSequenceStr}`);
    }

    // 开始预热
    async startWarmup() {
        if (this.isRunning) {
            console.log('⚠️ 预热已在进行中');
            return;
        }
        
        this.isRunning = true;
        console.log('🚀 开始推特养号预热（增强版 - 悬停卡片关注）...');
        console.log(`🎯 目标: 浏览${this.targets.tweetsToView}条推文, 点赞${this.targets.likesToGive}条, 关注${this.targets.followsToGive}个账号, 转发${this.targets.retweetsToGive}条`);
        console.log(`⏰ 预计时长: ${Math.round(this.duration / 60000)}分钟`);
        
        const startTime = Date.now();
        
        while (Date.now() - startTime < this.duration && this.isRunning) {
            if (!this.checkPageNavigation()) {
                console.log('🚨 检测到页面跳转，尝试返回推特主页...');
                const returnSuccess = await this.safeReturnToTwitter();
                if (!returnSuccess) {
                    console.log('❌ 无法返回推特主页，停止脚本');
                    break;
                }
                await this.delay(2000);
            }
            
            const action = this.selectNextAction();
            
            try {
                switch (action) {
                    case 'scroll':
                        await this.scrollAndBrowse();
                        break;
                    case 'hoverFollow':
                        await this.hoverCardFollow();
                        break;
                }
                
                this.showProgress();
                
                const waitTime = this.calculateWaitTime(action);
                await this.delayWithCountdown(waitTime, '浏览预热等待');
                
            } catch (error) {
                console.error('预热动作失败:', error);
                
                if (!this.checkPageNavigation()) {
                    console.log('🚨 检测到页面跳转，尝试返回推特主页...');
                    const returnSuccess = await this.safeReturnToTwitter();
                    if (returnSuccess) {
                        console.log('✅ 已返回推特主页，继续运行...');
                        await this.delay(2000);
                    } else {
                        console.log('❌ 无法返回推特主页，停止脚本');
                        break;
                    }
                }
            }
        }
        
        this.showFinalStats();
        this.isRunning = false;
    }

    // 停止预热
    stopWarmup() {
        this.isRunning = false;
        console.log('⏹️ 预热已停止');
    }
}

// 创建实例
const warmupBot = new TwitterWarmupBot();

// 全局脚本状态管理
window.twitterBot = {
    instance: warmupBot,
    isRunning: false,
    startTime: null,
    
    start() {
        console.log('🚀 启动推特养号脚本...');
        this.isRunning = true;
        this.startTime = Date.now();
        warmupBot.startWarmup();
    },
    
    stop() {
        console.log('⏹️ 停止推特养号脚本...');
        this.isRunning = false;
        warmupBot.stopWarmup();
    },
    
    status() {
        console.log('📊 脚本状态:', {
            isRunning: this.isRunning,
            botRunning: warmupBot.isRunning,
            currentUrl: window.location.href,
            startTime: this.startTime
        });
    }
};

// 页面跳转检测和自动恢复
function checkPageNavigation() {
    const currentUrl = window.location.href;
    const isTwitterHome = currentUrl.includes('twitter.com/home') || currentUrl.includes('x.com/home');
    const isUserProfile = currentUrl.includes('twitter.com/') && currentUrl.includes('/') && !currentUrl.includes('/home');
    
    if (isUserProfile) {
        console.log('🎯 检测到进入用户主页，开始关注流程...');
        console.log('📍 当前页面:', currentUrl);
        
        setTimeout(async () => {
            try {
                await warmupBot.continueFollowProcess();
            } catch (error) {
                console.error('关注流程失败:', error);
                try {
                    window.history.back();
                } catch (e) {
                    console.log('返回失败:', e);
                }
            }
        }, 2000);
    } else if (isTwitterHome) {
        console.log('🏠 检测到返回推特主页，继续预热...');
        if (window.twitterBot.isRunning) {
            console.log('✅ 脚本继续运行');
        } else {
            console.log('🔄 重新启动脚本...');
            window.twitterBot.start();
        }
    }
}

// 监听页面变化
window.addEventListener('popstate', checkPageNavigation);
window.addEventListener('pushstate', checkPageNavigation);
window.addEventListener('replacestate', checkPageNavigation);

// 监听URL变化
let currentUrl = window.location.href;
setInterval(() => {
    if (window.location.href !== currentUrl) {
        currentUrl = window.location.href;
        console.log('🔄 检测到URL变化:', currentUrl);
        checkPageNavigation();
    }
}, 1000);

// 页面加载完成后检查
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', checkPageNavigation);
} else {
    checkPageNavigation();
}

console.log('🐦 推特养号助手 - 修复版（页面跳转自动恢复）');
console.log('📋 使用方法:');
console.log('   twitterBot.start() - 启动脚本');
console.log('   twitterBot.stop() - 停止脚本');
console.log('   twitterBot.status() - 检查脚本状态');
console.log('   warmupBot.startWarmup() - 开始预热');
console.log('   warmupBot.stopWarmup() - 停止预热');
console.log('   warmupBot.showProgress() - 显示进度');

// 自动开始预热
warmupBot.startWarmup();
