// 增强版推特养号脚本 - 基于元素结构分析优化
class TwitterWarmupBot {
    constructor() {
        this.isRunning = false;
        this.stats = {
            tweetsViewed: 0,
            likesGiven: 0,
            followsGiven: 0,
            retweetsGiven: 0,
            hoverCardsTriggered: 0,
            startTime: Date.now(),
            actionSequence: [] // 记录动作序列
        };
        this.targets = {
            tweetsToView: 60 + Math.floor(Math.random() * 41), // 60-100条
            likesToGive: 0 + Math.floor(Math.random() * 0),     // 5-8条
            followsToGive: 3 + Math.floor(Math.random() * 3),   // 3-5个
            retweetsToGive: 0 + Math.floor(Math.random() * 0)   // 2-4条转发
        };
        this.duration = (20 + Math.random() * 10) * 60 * 1000; // 20-30分钟
    }

    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    async delayWithCountdown(ms, message) {
        const totalSeconds = Math.ceil(ms / 1000);
        let remainingSeconds = totalSeconds;
        
        console.log(`⏳ ${message} ${remainingSeconds}秒...`);
        
        const countdownInterval = setInterval(() => {
            remainingSeconds--;
            if (remainingSeconds > 0) {
                console.log(`⏳ ${message} ${remainingSeconds}秒...`);
            }
        }, 1000);
        
        await this.delay(ms);
        clearInterval(countdownInterval);
        console.log(`✅ ${message} 完成`);
    }

    async simulateHumanClick(element) {
        const rect = element.getBoundingClientRect();
        const x = rect.left + rect.width / 2 + (Math.random() - 0.5) * 10;
        const y = rect.top + rect.height / 2 + (Math.random() - 0.5) * 10;
        
        const mouseMoveEvent = new MouseEvent('mousemove', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(mouseMoveEvent);
        
        await this.delay(100 + Math.random() * 200);
        
        const clickEvent = new MouseEvent('click', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(clickEvent);
    }

    // 模拟鼠标悬停事件
    async simulateHover(element) {
        const rect = element.getBoundingClientRect();
        const x = rect.left + rect.width / 2;
        const y = rect.top + rect.height / 2;
        
        // 鼠标进入事件
        const mouseEnterEvent = new MouseEvent('mouseenter', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(mouseEnterEvent);
        
        // 鼠标悬停事件
        const mouseOverEvent = new MouseEvent('mouseover', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(mouseOverEvent);
        
        console.log('🖱️ 已触发悬停事件');
    }

    // 检查是否意外跳转页面
    checkPageNavigation() {
        const currentUrl = window.location.href;
        const isTwitterHome = currentUrl.includes('twitter.com') || currentUrl.includes('x.com');
        
        if (!isTwitterHome) {
            console.log('⚠️ 检测到页面跳转，当前URL:', currentUrl);
            return false;
        }
        return true;
    }

    // 安全返回推特主页
    async safeReturnToTwitter() {
        try {
            console.log('🔙 尝试返回推特主页...');
            
            // 尝试返回上一页
            if (window.history.length > 1) {
                window.history.back();
                await this.delay(2000);
                
                // 检查是否成功返回
                if (this.checkPageNavigation()) {
                    console.log('✅ 成功返回推特主页');
                    return true;
                }
            }
            
            // 如果返回失败，直接导航到推特主页
            console.log('🔙 返回失败，直接导航到推特主页...');
            window.location.href = 'https://twitter.com/home';
            await this.delay(3000);
            
            return this.checkPageNavigation();
        } catch (error) {
            console.error('返回推特主页失败:', error);
            return false;
        }
    }

    isElementInViewport(element) {
        const rect = element.getBoundingClientRect();
        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
            rect.right <= (window.innerWidth || document.documentElement.clientWidth)
        );
    }

    // 自然浏览行为
    async scrollAndBrowse() {
        try {
            // 增加滑动距离：300-600px
            const scrollDistance = Math.random() * 300 + 300;
            
            window.scrollBy({
                top: scrollDistance,
                behavior: 'smooth'
            });
            
            console.log(`📜 向下滑动 ${Math.round(scrollDistance)}px，浏览内容...`);
            this.stats.tweetsViewed++;
            
            const isInterested = Math.random() < 0.3;
            
            if (isInterested) {
                const readingTime = Math.random() * 4 + 2;
                console.log(`👀 发现感兴趣的内容！阅读时间: ${readingTime.toFixed(1)}秒`);
                await this.delayWithCountdown(readingTime * 1000, '阅读感兴趣的内容');
                
                const continueDelay = Math.random() * 1000 + 500;
                await this.delayWithCountdown(continueDelay, '阅读完成，等待继续浏览');
            } else {
                const skipTime = Math.random() * 1.5 + 0.5;
                console.log(`⏭️ 内容不感兴趣，快速跳过 (${skipTime.toFixed(1)}秒)`);
                await this.delayWithCountdown(skipTime * 1000, '快速跳过不感兴趣内容');
                
                const skipDelay = Math.random() * 500 + 200;
                await this.delayWithCountdown(skipDelay, '跳过完成，等待继续浏览');
            }
            
            // 增加回看概率：从3%提升到8%
            if (Math.random() < 0.08) {
                const backScroll = 50 + Math.random() * 100; // 增加回看距离
                window.scrollBy({
                    top: -backScroll,
                    behavior: 'smooth'
                });
                console.log(`↩️ 回看内容 ${Math.round(backScroll)}px`);
                
                const backScrollDelay = 1000 + Math.random() * 2000;
                await this.delayWithCountdown(backScrollDelay, '回看完成，等待继续');
            }
            
            return true;
        } catch (error) {
            console.error('自然浏览失败:', error);
            return false;
        }
    }

    // 点赞推文
    async likeTweet() {
        try {
            const likeButtons = document.querySelectorAll('[data-testid="like"]');
            const visibleLikes = Array.from(likeButtons).filter(btn => 
                btn.offsetParent !== null && 
                !btn.closest('[aria-label*="Liked"]') &&
                this.isElementInViewport(btn)
            );
            
            if (visibleLikes.length > 0) {
                const randomButton = visibleLikes[Math.floor(Math.random() * visibleLikes.length)];
                
                randomButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
                await this.delay(800 + Math.random() * 1200);
                
                await this.simulateHumanClick(randomButton);
                console.log('👍 已点赞推文');
                this.stats.likesGiven++;
                
                await this.delayWithCountdown(2000 + Math.random() * 3000, '点赞后停留');
                return true;
            } else {
                console.log('🔍 未找到可点赞的推文');
                return false;
            }
        } catch (error) {
            console.error('点赞失败:', error);
            return false;
        }
    }

    // 转发推文
    async retweetTweet() {
        try {
            console.log('🔍 开始查找转发按钮...');
            
            const retweetSelectors = [
                '[data-testid="retweet"]',
                '[aria-label*="Retweet"]',
                '[aria-label*="转发"]',
                'button[data-testid="retweet"]'
            ];
            
            let retweetButtons = [];
            for (const selector of retweetSelectors) {
                const buttons = document.querySelectorAll(selector);
                retweetButtons = retweetButtons.concat(Array.from(buttons));
            }
            
            const visibleRetweets = retweetButtons.filter(btn => {
                const isVisible = btn.offsetParent !== null;
                const isInViewport = this.isElementInViewport(btn);
                const isNotRetweeted = !btn.closest('[aria-label*="Retweeted"]') && 
                                     !btn.closest('[aria-label*="已转发"]');
                
                return isVisible && isInViewport && isNotRetweeted;
            });
            
            console.log(`📊 找到 ${visibleRetweets.length} 个可转发的按钮`);
            
            if (visibleRetweets.length === 0) {
                console.log('❌ 未找到可转发的推文');
                return false;
            }
            
            const randomButton = visibleRetweets[Math.floor(Math.random() * visibleRetweets.length)];
            console.log('🎯 选择转发按钮:', {
                ariaLabel: randomButton.getAttribute('aria-label'),
                dataTestId: randomButton.getAttribute('data-testid')
            });
            
            randomButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(1000 + Math.random() * 1000);
            
            await this.simulateHumanClick(randomButton);
            console.log('🔄 已点击转发按钮');
            
            await this.delay(1500 + Math.random() * 1000);
            
            const confirmSelectors = [
                '[data-testid="retweetConfirm"]',
                '[aria-label*="Retweet"]',
                'button[data-testid="retweetConfirm"]',
                'div[role="button"][data-testid="retweetConfirm"]'
            ];
            
            let confirmButton = null;
            for (const selector of confirmSelectors) {
                const button = document.querySelector(selector);
                if (button && button.offsetParent !== null) {
                    confirmButton = button;
                    break;
                }
            }
            
            if (confirmButton) {
                await this.simulateHumanClick(confirmButton);
                console.log('✅ 已确认转发');
                this.stats.retweetsGiven++;
                
                await this.delayWithCountdown(3000 + Math.random() * 2000, '转发后停留');
                return true;
            } else {
                console.log('⚠️ 未找到确认转发按钮，可能已直接转发');
                this.stats.retweetsGiven++;
                return true;
            }
            
        } catch (error) {
            console.error('转发失败:', error);
            return false;
        }
    }

    // 点击头像进入主页关注功能
    async hoverCardFollow() {
        try {
            console.log('🔍 开始点击头像进入主页关注流程...');
            
            // 更广泛的头像选择器，确保能找到推文中的头像
            const avatarSelectors = [
                // 标准推特头像选择器
                '[data-testid="UserAvatar-Container"]',
                '[data-testid="UserAvatar"]',
                // 推文区域内的头像
                '[data-testid="primaryColumn"] [data-testid="UserAvatar-Container"]',
                '[data-testid="primaryColumn"] [data-testid="UserAvatar"]',
                // 文章内的头像
                'article [data-testid="UserAvatar-Container"]',
                'article [data-testid="UserAvatar"]',
                // 通用头像选择器
                'div[role="button"][aria-label*="头像"]',
                'div[role="button"][aria-label*="Avatar"]',
                'div[role="button"][aria-label*="Profile"]',
                // CSS类名选择器
                'div[class*="css-175oi2r"][class*="r-172uzmj"]',
                'div[class*="UserAvatar"]',
                'div[role="button"][class*="avatar"]',
                // 更广泛的头像选择器
                'div[class*="css-175oi2r"] img[alt*="头像"]',
                'div[class*="css-175oi2r"] img[alt*="Avatar"]',
                'div[class*="css-175oi2r"] img[alt*="Profile"]'
            ];
            
            let avatarElements = [];
            for (const selector of avatarSelectors) {
                const elements = document.querySelectorAll(selector);
                avatarElements = avatarElements.concat(Array.from(elements));
            }
            
            // 先进行基础过滤，确保能找到头像
            const basicAvatars = avatarElements.filter(avatar => {
                const isVisible = avatar.offsetParent !== null;
                const hasSize = avatar.offsetWidth > 0 && avatar.offsetHeight > 0;
                return isVisible && hasSize;
            });
            
            console.log(`📊 基础过滤后找到 ${basicAvatars.length} 个头像`);
            
            // 进一步过滤，避免导航栏头像
            const visibleAvatars = basicAvatars.filter(avatar => {
                // 避免点击导航栏中的头像（左侧边栏）
                const isNotSidebar = !avatar.closest('[data-testid="sidebarColumn"]') && 
                                   !avatar.closest('[role="navigation"]') &&
                                   !avatar.closest('nav') &&
                                   !avatar.closest('[data-testid="SideNav"]');
                
                // 确保是推文区域的头像
                const isInTweetArea = avatar.closest('[data-testid="primaryColumn"]') ||
                                     avatar.closest('[data-testid="tweet"]') ||
                                     avatar.closest('article') ||
                                     avatar.closest('[role="article"]');
                
                return isNotSidebar && isInTweetArea;
            });
            
            console.log(`📊 找到 ${visibleAvatars.length} 个可见可点击头像`);
            
            if (visibleAvatars.length === 0) {
                console.log('❌ 未找到任何头像元素，尝试备用方法...');
                
                // 备用方法：使用更简单的选择器
                const fallbackSelectors = [
                    'img[alt*="头像"]',
                    'img[alt*="Avatar"]',
                    'img[alt*="Profile"]',
                    'div[role="button"] img',
                    'article img[alt]',
                    '[data-testid="primaryColumn"] img[alt]'
                ];
                
                let fallbackAvatars = [];
                for (const selector of fallbackSelectors) {
                    const elements = document.querySelectorAll(selector);
                    fallbackAvatars = fallbackAvatars.concat(Array.from(elements));
                }
                
                // 过滤可见的头像
                const visibleFallbackAvatars = fallbackAvatars.filter(avatar => {
                    const isVisible = avatar.offsetParent !== null;
                    const hasSize = avatar.offsetWidth > 0 && avatar.offsetHeight > 0;
                    const isNotSidebar = !avatar.closest('[data-testid="sidebarColumn"]') && 
                                       !avatar.closest('[role="navigation"]') &&
                                       !avatar.closest('nav') &&
                                       !avatar.closest('[data-testid="SideNav"]');
                    const isInTweetArea = avatar.closest('[data-testid="primaryColumn"]') ||
                                       avatar.closest('[data-testid="tweet"]') ||
                                       avatar.closest('article') ||
                                       avatar.closest('[role="article"]');
                    
                    return isVisible && hasSize && isNotSidebar && isInTweetArea;
                });
                
                console.log(`📊 备用方法找到 ${visibleFallbackAvatars.length} 个头像`);
                
                if (visibleFallbackAvatars.length === 0) {
                    console.log('❌ 备用方法也未找到头像元素');
                    return false;
                }
                
                // 使用备用方法找到的头像
                const randomAvatar = visibleFallbackAvatars[Math.floor(Math.random() * visibleFallbackAvatars.length)];
                console.log('🎯 使用备用方法选择的头像:', {
                    tagName: randomAvatar.tagName,
                    alt: randomAvatar.getAttribute('alt'),
                    className: randomAvatar.className
                });
                
                // 点击头像
                await this.simulateHumanClick(randomAvatar);
                console.log('✅ 已点击头像，进入用户主页');
                
                // 等待页面加载
                await this.delayWithCountdown(2000, '等待用户主页加载');
                
                // 继续后续流程...
                return await this.continueFollowProcess();
            }
            
            // 显示找到的头像位置信息
            visibleAvatars.forEach((avatar, index) => {
                const parentElement = avatar.closest('article') || avatar.closest('[data-testid="tweet"]') || avatar.closest('[role="article"]');
                const isInTweet = !!parentElement;
                console.log(`📍 头像${index + 1}: 在推文区域=${isInTweet}, 父元素=${parentElement?.tagName || '无'}`);
            });
            
            const randomAvatar = visibleAvatars[Math.floor(Math.random() * visibleAvatars.length)];
            const parentElement = randomAvatar.closest('article') || randomAvatar.closest('[data-testid="tweet"]') || randomAvatar.closest('[role="article"]');
            
            console.log('🎯 选择的头像元素:', {
                className: randomAvatar.className,
                role: randomAvatar.getAttribute('role'),
                ariaLabel: randomAvatar.getAttribute('aria-label'),
                tagName: randomAvatar.tagName,
                parentElement: parentElement?.tagName || '无',
                isInTweet: !!parentElement
            });
            
            // 滚动到头像位置
            randomAvatar.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(1000);
            
            // 记录点击前的URL
            const beforeUrl = window.location.href;
            console.log('📍 点击前URL:', beforeUrl);
            
            // 点击头像进入主页
            await this.simulateHumanClick(randomAvatar);
            console.log('✅ 已点击头像，等待页面跳转...');
            
            // 等待页面加载和跳转
            await this.delayWithCountdown(3000, '等待页面跳转');
            
            // 检查是否成功跳转
            const afterUrl = window.location.href;
            console.log('📍 点击后URL:', afterUrl);
            
            if (beforeUrl === afterUrl) {
                console.log('⚠️ 页面未跳转，可能点击失败');
                return false;
            }
            
            console.log('✅ 成功进入用户主页');
            
            // 等待页面完全加载
            await this.delayWithCountdown(2000, '等待页面完全加载');
            
            // 检查页面是否还在加载
            if (document.readyState !== 'complete') {
                console.log('⏳ 页面仍在加载，等待完成...');
                await this.delayWithCountdown(3000, '等待页面加载完成');
            }
            
            // 继续关注流程
            console.log('🔄 开始关注流程...');
            return await this.continueFollowProcess();
            
        } catch (error) {
            console.error('点击头像关注失败:', error);
            
            // 出错时也尝试返回
            try {
                console.log('🔄 尝试返回原页面...');
                window.history.back();
                await this.delay(2000);
                console.log('✅ 已返回原页面');
            } catch (e) {
                console.log('❌ 返回失败:', e);
            }
            
            return false;
        }
    }

    // 继续关注流程的通用方法
    async continueFollowProcess() {
        try {
            // 检查当前页面状态
            const currentUrl = window.location.href;
            console.log('📍 当前页面URL:', currentUrl);
            
            // 在主页浏览内容 - 模拟真实用户行为
            console.log('📜 在用户主页浏览内容...');
            
            // 随机浏览2-4次，减少浏览时间
            const browseCount = 2 + Math.floor(Math.random() * 3); // 2-4次滚动
            
            for (let i = 0; i < browseCount; i++) {
                // 随机滚动距离：200-600px
                const scrollDistance = Math.random() * 400 + 200;
                window.scrollBy({
                    top: scrollDistance,
                    behavior: 'smooth'
                });
                console.log(`📜 主页浏览滚动 ${Math.round(scrollDistance)}px (${i + 1}/${browseCount})`);
                
                // 模拟阅读时间：1-3秒
                const readingTime = Math.random() * 2 + 1;
                await this.delayWithCountdown(readingTime * 1000, '浏览用户内容');
            }
            
            // 查找关注按钮
            console.log('🔍 在用户主页查找关注按钮...');
            
            // 更广泛的关注按钮选择器
            const followSelectors = [
                // 标准选择器
                'button[data-testid="follow"]',
                'div[role="button"][data-testid="follow"]',
                // 文本内容选择器
                'button:contains("关注")',
                'div[role="button"]:contains("关注")',
                'button:contains("Follow")',
                'div[role="button"]:contains("Follow")',
                // aria-label选择器
                'button[aria-label*="关注"]',
                'div[role="button"][aria-label*="关注"]',
                'button[aria-label*="Follow"]',
                'div[role="button"][aria-label*="Follow"]',
                // 更广泛的选择器
                'button[class*="follow"]',
                'div[role="button"][class*="follow"]',
                'button[class*="Follow"]',
                'div[role="button"][class*="Follow"]'
            ];
            
            let followButton = null;
            let foundWithSelector = '';
            
            // 尝试所有选择器
            for (const selector of followSelectors) {
                try {
                    const buttons = document.querySelectorAll(selector);
                    console.log(`🔍 选择器 "${selector}" 找到 ${buttons.length} 个按钮`);
                    
                    for (const button of buttons) {
                        if (button.offsetParent !== null) {
                            const text = button.textContent?.trim() || '';
                            const ariaLabel = button.getAttribute('aria-label') || '';
                            
                            console.log(`🔍 检查按钮: text="${text}", aria-label="${ariaLabel}"`);
                            
                            if (text.includes('关注') || text.includes('Follow') || 
                                ariaLabel.includes('关注') || ariaLabel.includes('Follow')) {
                                followButton = button;
                                foundWithSelector = selector;
                                console.log(`✅ 找到关注按钮，使用选择器: ${selector}`);
                                break;
                            }
                        }
                    }
                    if (followButton) break;
                } catch (e) {
                    console.log(`❌ 选择器 "${selector}" 出错:`, e.message);
                    continue;
                }
            }
            
            // 如果没找到，使用更广泛的搜索
            if (!followButton) {
                console.log('🔍 使用更广泛的搜索策略...');
                const allButtons = document.querySelectorAll('button, div[role="button"]');
                console.log(`🔍 页面总共有 ${allButtons.length} 个按钮`);
                
                for (const button of allButtons) {
                    const text = button.textContent?.trim() || '';
                    const ariaLabel = button.getAttribute('aria-label') || '';
                    
                    if ((text === '关注' || text === 'Follow' || 
                         ariaLabel.includes('关注') || ariaLabel.includes('Follow')) &&
                        button.offsetParent !== null) {
                        followButton = button;
                        foundWithSelector = '广泛搜索';
                        console.log(`✅ 广泛搜索找到关注按钮: text="${text}", aria-label="${ariaLabel}"`);
                        break;
                    }
                }
            }
            
            if (followButton) {
                console.log('✅ 在用户主页找到关注按钮:', {
                    textContent: followButton.textContent?.trim(),
                    ariaLabel: followButton.getAttribute('aria-label'),
                    className: followButton.className,
                    foundWith: foundWithSelector
                });
                
                // 滚动到按钮位置
                followButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
                await this.delay(500);
                
                // 点击关注按钮
                await this.simulateHumanClick(followButton);
                console.log('✅ 已点击关注按钮');
                
                this.stats.followsGiven++;
                
                // 等待关注操作完成
                await this.delayWithCountdown(2000, '关注操作完成');
                
                // 返回上一页
                console.log('🔙 返回上一页...');
                window.history.back();
                await this.delayWithCountdown(2000, '等待返回完成');
                
                console.log('✅ 关注流程完成，已返回');
                return true;
            } else {
                console.log('❌ 在用户主页未找到关注按钮');
                
                // 显示页面上的所有按钮信息用于调试
                const allButtons = document.querySelectorAll('button, div[role="button"]');
                console.log('🔍 页面上的所有按钮:');
                allButtons.forEach((button, index) => {
                    if (index < 10) { // 只显示前10个
                        console.log(`  按钮${index + 1}: text="${button.textContent?.trim()}", aria-label="${button.getAttribute('aria-label')}", visible=${button.offsetParent !== null}`);
                    }
                });
                
                // 返回上一页
                console.log('🔙 未找到关注按钮，返回上一页...');
                window.history.back();
                await this.delay(1000);
                
                return false;
            }
            
        } catch (error) {
            console.error('继续关注流程失败:', error);
            
            // 出错时也尝试返回
            try {
                window.history.back();
            } catch (e) {
                console.log('返回失败:', e);
            }
            
            return false;
        }
    }

    // 传统关注用户方法（保留作为备用）
    async followUser() {
        try {
            console.log('🔍 开始查找Follow按钮...');
            
            const allElements = document.querySelectorAll('*');
            const followButtons = Array.from(allElements).filter(el => 
                el.textContent && el.textContent.includes('Follow') && 
                el.offsetParent !== null &&
                el.tagName === 'BUTTON'
            );
            
            console.log(`📊 找到 ${followButtons.length} 个Follow按钮`);
            
            if (followButtons.length === 0) {
                console.log('❌ 未找到任何Follow按钮');
                return false;
            }
            
            const visibleFollows = followButtons.filter(btn => {
                const isVisible = btn.offsetParent !== null;
                const isInViewport = this.isElementInViewport(btn);
                const hasFollowText = btn.textContent && btn.textContent.includes('Follow');
                
                return isVisible && isInViewport && hasFollowText;
            });
            
            console.log(`📊 可见的Follow按钮: ${visibleFollows.length}个`);
            
            if (visibleFollows.length === 0) {
                console.log('❌ 没有可见的Follow按钮');
                return false;
            }
            
            const randomButton = visibleFollows[Math.floor(Math.random() * visibleFollows.length)];
            console.log('🎯 选择的Follow按钮:', {
                textContent: randomButton.textContent?.trim(),
                ariaLabel: randomButton.getAttribute('aria-label')
            });
            
            randomButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(1000);
            
            await this.simulateHumanClick(randomButton);
            console.log('✅ 已点击Follow按钮');
            
            await this.delay(2000);
            console.log('✅ 关注操作完成');
            this.stats.followsGiven++;
            
            return true;
        } catch (error) {
            console.error('关注失败:', error);
            return false;
        }
    }

    // 检查动作间隔，避免连续相同动作
    shouldAvoidAction(action) {
        const recentActions = this.stats.actionSequence.slice(-3); // 最近3个动作
        const sameActionCount = recentActions.filter(a => a === action).length;
        return sameActionCount >= 2; // 避免连续3次相同动作
    }

    // 检查是否需要强制互动
    shouldForceInteraction() {
        const totalInteractions = this.targets.likesToGive + this.targets.followsToGive + this.targets.retweetsToGive;
        const completedInteractions = this.stats.likesGiven + this.stats.followsGiven + this.stats.retweetsGiven;
        const elapsed = (Date.now() - this.stats.startTime) / 1000 / 60; // 分钟
        
        // 如果运行超过5分钟但互动进度低于20%，强制互动
        if (elapsed > 5 && (completedInteractions / totalInteractions) < 0.2) {
            return true;
        }
        
        // 如果连续10个动作都是滚动，强制互动
        const recentActions = this.stats.actionSequence.slice(-10);
        if (recentActions.length >= 10 && recentActions.every(action => action === 'scroll')) {
            return true;
        }
        
        return false;
    }

    // 智能计算等待时间
    calculateWaitTime(action) {
        let baseTime = 4000; // 基础4秒
        
        // 根据动作类型调整
        switch (action) {
            case 'scroll':
                baseTime = 1500 + Math.random() * 2000; // 1.5-3.5秒，减少滚动等待时间
                break;
            case 'like':
                baseTime = 2000 + Math.random() * 2000; // 2-4秒
                break;
            case 'hoverFollow':
            case 'follow':
                baseTime = 4000 + Math.random() * 4000; // 4-8秒
                break;
            case 'retweet':
                baseTime = 5000 + Math.random() * 3000; // 5-8秒
                break;
        }
        
        // 根据最近动作调整
        const recentActions = this.stats.actionSequence.slice(-2);
        if (recentActions.includes(action)) {
            baseTime *= 1.5; // 相同动作间隔更久
        }
        
        return Math.round(baseTime);
    }

    selectNextAction() {
        const actions = [];
        
        // 基础滚动行为 - 大幅增加滚动频率，保持常态化浏览
        for (let i = 0; i < 15; i++) {
            actions.push('scroll');
        }
        
        // 收集所有可用的互动选项
        const availableActions = [];
        
        // 点赞选项
        if (this.stats.likesGiven < this.targets.likesToGive && 
            !this.shouldAvoidAction('like')) {
            availableActions.push('like');
        }
        
        // 关注选项 - 智能选择关注方式
        if (this.stats.followsGiven < this.targets.followsToGive) {
            const followMethods = [];
            if (!this.shouldAvoidAction('hoverFollow')) {
                followMethods.push('hoverFollow');
            }
            if (!this.shouldAvoidAction('follow')) {
                followMethods.push('follow');
            }
            
            if (followMethods.length > 0) {
                // 随机选择关注方式
                const selectedMethod = followMethods[Math.floor(Math.random() * followMethods.length)];
                availableActions.push(selectedMethod);
            }
        }
        
        // 转发选项
        if (this.stats.retweetsGiven < this.targets.retweetsToGive && 
            !this.shouldAvoidAction('retweet')) {
            availableActions.push('retweet');
        }
        
        // 检查是否需要强制互动
        const forceInteraction = this.shouldForceInteraction();
        
        if (forceInteraction && availableActions.length > 0) {
            // 强制互动模式：只选择互动动作
            console.log('🚨 强制互动模式激活！');
            availableActions.forEach(action => {
                actions.push(action);
                actions.push(action);
                actions.push(action); // 大幅增加权重
            });
        } else {
            // 智能权重分配 - 根据完成度调整权重
            availableActions.forEach(action => {
                let weight = 1;
                
                // 根据完成度调整权重
                switch (action) {
                    case 'like':
                        const likeProgress = this.stats.likesGiven / this.targets.likesToGive;
                        if (likeProgress >= 0.8) weight = 0.5; // 点赞接近完成时降低权重
                        else if (likeProgress >= 0.6) weight = 0.7;
                        break;
                    case 'hoverFollow':
                    case 'follow':
                        const followProgress = this.stats.followsGiven / this.targets.followsToGive;
                        if (followProgress < 0.2) weight = 2; // 关注刚开始时增加权重
                        else if (followProgress < 0.5) weight = 1.5;
                        break;
                    case 'retweet':
                        const retweetProgress = this.stats.retweetsGiven / this.targets.retweetsToGive;
                        if (retweetProgress < 0.3) weight = 2; // 转发刚开始时增加权重
                        else if (retweetProgress < 0.6) weight = 1.5;
                        break;
                }
                
                // 添加权重，但限制互动动作数量，保持滚动为主
                const maxInteractions = Math.min(Math.round(weight * 2), 3); // 最多3个互动动作
                for (let i = 0; i < maxInteractions; i++) {
                    actions.push(action);
                }
            });
        }
        
        // 如果所有目标都完成了，增加更多滚动
        if (this.stats.likesGiven >= this.targets.likesToGive && 
            this.stats.followsGiven >= this.targets.followsToGive && 
            this.stats.retweetsGiven >= this.targets.retweetsToGive) {
            for (let i = 0; i < 8; i++) {
                actions.push('scroll');
            }
        }
        
        // 检查是否需要强制平衡
        const likeProgress = this.stats.likesGiven / this.targets.likesToGive;
        const followProgress = this.stats.followsGiven / this.targets.followsToGive;
        const retweetProgress = this.stats.retweetsGiven / this.targets.retweetsToGive;
        
        let selectedAction;
        
        // 如果点赞完成度超过80%，强制选择其他动作
        if (likeProgress >= 0.8 && (followProgress < 0.5 || retweetProgress < 0.5)) {
            const nonLikeActions = actions.filter(action => action !== 'like' && action !== 'scroll');
            if (nonLikeActions.length > 0) {
                selectedAction = nonLikeActions[Math.floor(Math.random() * nonLikeActions.length)];
                console.log(`⚖️ 强制平衡：点赞完成度${Math.round(likeProgress*100)}%，选择其他动作: ${selectedAction}`);
            } else {
                selectedAction = actions[Math.floor(Math.random() * actions.length)];
            }
        } else {
            selectedAction = actions[Math.floor(Math.random() * actions.length)];
        }
        
        // 调试信息 - 显示选择过程
        if (this.stats.actionSequence.length % 5 === 0) { // 每5个动作显示一次调试信息
            const scrollCount = actions.filter(a => a === 'scroll').length;
            const likeCount = actions.filter(a => a === 'like').length;
            const followCount = actions.filter(a => a === 'hoverFollow' || a === 'follow').length;
            const retweetCount = actions.filter(a => a === 'retweet').length;
            
            // 计算滚动比例
            const scrollRatio = Math.round((scrollCount / actions.length) * 100);
            
            console.log(`🔍 动作选择调试: 总动作数=${actions.length}, 滚动=${scrollCount}(${scrollRatio}%), 点赞=${likeCount}, 关注=${followCount}, 转发=${retweetCount}`);
            console.log(`🎯 可用互动选项:`, availableActions);
            console.log(`📊 当前统计: 点赞${this.stats.likesGiven}/${this.targets.likesToGive}(${Math.round(this.stats.likesGiven/this.targets.likesToGive*100)}%), 关注${this.stats.followsGiven}/${this.targets.followsToGive}(${Math.round(this.stats.followsGiven/this.targets.followsToGive*100)}%), 转发${this.stats.retweetsGiven}/${this.targets.retweetsToGive}(${Math.round(this.stats.retweetsGiven/this.targets.retweetsToGive*100)}%)`);
            
            // 检查滚动频率是否足够
            if (scrollRatio < 60) {
                console.log('⚠️ 滚动频率偏低，建议增加滚动动作');
            }
        }
        
        // 记录动作序列
        this.stats.actionSequence = this.stats.actionSequence || [];
        this.stats.actionSequence.push(selectedAction);
        if (this.stats.actionSequence.length > 10) {
            this.stats.actionSequence.shift(); // 保持最近10个动作
        }
        
        return selectedAction;
    }

    showProgress() {
        const elapsed = Math.round((Date.now() - this.stats.startTime) / 1000);
        const remaining = Math.max(0, Math.round((this.duration - (Date.now() - this.stats.startTime)) / 1000));
        const totalInteractions = this.targets.likesToGive + this.targets.followsToGive + this.targets.retweetsToGive;
        const completedInteractions = this.stats.likesGiven + this.stats.followsGiven + this.stats.retweetsGiven;
        const interactionProgress = Math.round((completedInteractions / totalInteractions) * 100);
        
        // 显示最近动作序列
        const recentActions = this.stats.actionSequence.slice(-3).join(' → ');
        
        console.log(`📊 预热进度: ${elapsed}秒 | 推文:${this.stats.tweetsViewed}/${this.targets.tweetsToView} | 点赞:${this.stats.likesGiven}/${this.targets.likesToGive} | 关注:${this.stats.followsGiven}/${this.targets.followsToGive} | 转发:${this.stats.retweetsGiven}/${this.targets.retweetsToGive} | 悬停:${this.stats.hoverCardsTriggered} | 互动进度:${interactionProgress}% | 最近动作:${recentActions} | 剩余:${remaining}秒`);
    }

    async startWarmup() {
        if (this.isRunning) {
            console.log('⚠️ 预热已在进行中');
            return;
        }
        
        this.isRunning = true;
        console.log('🚀 开始推特养号预热（增强版 - 悬停卡片关注）...');
        console.log(`🎯 目标: 浏览${this.targets.tweetsToView}条推文, 点赞${this.targets.likesToGive}条, 关注${this.targets.followsToGive}个账号, 转发${this.targets.retweetsToGive}条`);
        console.log(`⏰ 预计时长: ${Math.round(this.duration / 60000)}分钟`);
        
        const startTime = Date.now();
        
        while (Date.now() - startTime < this.duration && this.isRunning) {
            // 检查是否意外跳转页面
            if (!this.checkPageNavigation()) {
                console.log('🚨 检测到页面跳转，尝试返回推特主页...');
                const returnSuccess = await this.safeReturnToTwitter();
                if (!returnSuccess) {
                    console.log('❌ 无法返回推特主页，停止脚本');
                    break;
                }
                // 返回成功后继续
                await this.delay(2000);
            }
            
            const action = this.selectNextAction();
            
            try {
                switch (action) {
                    case 'scroll':
                        await this.scrollAndBrowse();
                        break;
                    case 'like':
                        await this.likeTweet();
                        break;
                    case 'hoverFollow':
                        await this.hoverCardFollow();
                        break;
                    case 'follow':
                        await this.followUser();
                        break;
                    case 'retweet':
                        await this.retweetTweet();
                        break;
                }
                
                this.showProgress();
                
                // 智能调整等待时间
                const waitTime = this.calculateWaitTime(action);
                await this.delayWithCountdown(waitTime, '浏览预热等待');
                
            } catch (error) {
                console.error('预热动作失败:', error);
                
                // 检查是否在错误的页面
                if (!this.checkPageNavigation()) {
                    console.log('🚨 检测到页面跳转，尝试返回推特主页...');
                    const returnSuccess = await this.safeReturnToTwitter();
                    if (returnSuccess) {
                        console.log('✅ 已返回推特主页，继续运行...');
                        await this.delay(2000);
                    } else {
                        console.log('❌ 无法返回推特主页，停止脚本');
                        break;
                    }
                }
            }
        }
        
        this.showFinalStats();
        this.isRunning = false;
    }

    showFinalStats() {
        const totalTime = Math.round((Date.now() - this.stats.startTime) / 1000);
        
        console.log('✅ 浏览预热完成!');
        console.log(`📈 最终统计:`);
        console.log(`   总时间: ${totalTime}秒 (${Math.round(totalTime/60)}分钟)`);
        console.log(`   浏览推文: ${this.stats.tweetsViewed}条`);
        console.log(`   点赞: ${this.stats.likesGiven}条`);
        console.log(`   关注: ${this.stats.followsGiven}个账号`);
        console.log(`   转发: ${this.stats.retweetsGiven}条`);
        console.log(`   触发悬停卡片: ${this.stats.hoverCardsTriggered}次`);
        
        const tweetProgress = Math.round(this.stats.tweetsViewed/this.targets.tweetsToView*100);
        const likeProgress = Math.round(this.stats.likesGiven/this.targets.likesToGive*100);
        const followProgress = Math.round(this.stats.followsGiven/this.targets.followsToGive*100);
        const retweetProgress = Math.round(this.stats.retweetsGiven/this.targets.retweetsToGive*100);
        
        console.log(`   完成度: 推文${tweetProgress}% | 点赞${likeProgress}% | 关注${followProgress}% | 转发${retweetProgress}%`);
        
        // 显示动作分布统计
        const actionCounts = {};
        this.stats.actionSequence.forEach(action => {
            actionCounts[action] = (actionCounts[action] || 0) + 1;
        });
        console.log(`   动作分布:`, actionCounts);
        
        // 显示动作序列示例
        const actionSequenceStr = this.stats.actionSequence.slice(-10).join(' → ');
        console.log(`   最近动作序列: ${actionSequenceStr}`);
    }

    stopWarmup() {
        this.isRunning = false;
        console.log('⏹️ 预热已停止');
    }

    // 手动测试互动功能
    async testInteractions() {
        console.log('🧪 开始测试互动功能...');
        
        console.log('🔍 测试点赞功能...');
        const likeResult = await this.likeTweet();
        console.log('点赞结果:', likeResult ? '成功' : '失败');
        
        await this.delay(2000);
        
        console.log('🔍 测试悬停关注功能...');
        const hoverFollowResult = await this.hoverCardFollow();
        console.log('悬停关注结果:', hoverFollowResult ? '成功' : '失败');
        
        await this.delay(2000);
        
        console.log('🔍 测试转发功能...');
        const retweetResult = await this.retweetTweet();
        console.log('转发结果:', retweetResult ? '成功' : '失败');
        
        console.log('✅ 互动功能测试完成');
    }

    // 强制进行一次互动
    async forceInteraction() {
        console.log('🚨 强制进行互动...');
        
        const availableActions = [];
        
        if (this.stats.likesGiven < this.targets.likesToGive) {
            availableActions.push('like');
        }
        if (this.stats.followsGiven < this.targets.followsToGive) {
            availableActions.push('hoverFollow');
        }
        if (this.stats.retweetsGiven < this.targets.retweetsToGive) {
            availableActions.push('retweet');
        }
        
        if (availableActions.length > 0) {
            const action = availableActions[Math.floor(Math.random() * availableActions.length)];
            console.log(`🎯 强制执行动作: ${action}`);
            
            switch (action) {
                case 'like':
                    return await this.likeTweet();
                case 'hoverFollow':
                    return await this.hoverCardFollow();
                case 'retweet':
                    return await this.retweetTweet();
            }
        } else {
            console.log('❌ 没有可用的互动选项');
            return false;
        }
    }

    // 手动返回推特主页
    async returnToTwitter() {
        console.log('🔙 手动返回推特主页...');
        return await this.safeReturnToTwitter();
    }

    // 检查当前页面状态
    checkCurrentPage() {
        const currentUrl = window.location.href;
        const isTwitter = currentUrl.includes('twitter.com') || currentUrl.includes('x.com');
        
        console.log(`📍 当前页面: ${currentUrl}`);
        console.log(`✅ 是否在推特: ${isTwitter ? '是' : '否'}`);
        
        return isTwitter;
    }

    // 手动恢复脚本运行
    async recoverScript() {
        try {
            console.log('🔄 手动恢复脚本运行...');
            
            // 检查当前页面
            if (!this.checkPageNavigation()) {
                console.log('🚨 不在推特页面，尝试返回...');
                const returnSuccess = await this.safeReturnToTwitter();
                if (!returnSuccess) {
                    console.log('❌ 无法返回推特主页');
                    return false;
                }
            }
            
            // 检查脚本是否还在运行
            if (!this.isRunning) {
                console.log('🔄 重新启动脚本...');
                this.isRunning = true;
                // 不重新开始整个流程，只是恢复运行状态
                console.log('✅ 脚本已恢复运行状态');
                return true;
            } else {
                console.log('✅ 脚本仍在运行中');
                return true;
            }
        } catch (error) {
            console.error('恢复脚本失败:', error);
            return false;
        }
    }
}

// 创建实例
const warmupBot = new TwitterWarmupBot();

// 全局脚本状态管理
window.twitterBot = {
    instance: warmupBot,
    isRunning: false,
    startTime: null,
    
    // 启动脚本
    start() {
        console.log('🚀 启动推特养号脚本...');
        this.isRunning = true;
        this.startTime = Date.now();
        warmupBot.startWarmup();
    },
    
    // 停止脚本
    stop() {
        console.log('⏹️ 停止推特养号脚本...');
        this.isRunning = false;
        warmupBot.stopWarmup();
    },
    
    // 检查状态
    status() {
        console.log('📊 脚本状态:', {
            isRunning: this.isRunning,
            botRunning: warmupBot.isRunning,
            currentUrl: window.location.href,
            startTime: this.startTime
        });
    }
};

console.log('🐦 推特养号助手 - 增强版（页面跳转自动恢复）');
console.log('📋 使用方法:');
console.log('   twitterBot.start() - 启动脚本');
console.log('   twitterBot.stop() - 停止脚本');
console.log('   twitterBot.status() - 检查脚本状态');
console.log('   warmupBot.startWarmup() - 开始预热');
console.log('   warmupBot.stopWarmup() - 停止预热');
console.log('   warmupBot.showProgress() - 显示进度');
console.log('   warmupBot.testInteractions() - 测试所有互动功能');
console.log('   warmupBot.forceInteraction() - 强制进行一次互动');
console.log('   warmupBot.returnToTwitter() - 手动返回推特主页');
console.log('   warmupBot.checkCurrentPage() - 检查当前页面状态');
console.log('   warmupBot.recoverScript() - 手动恢复脚本运行');
console.log('   warmupBot.hoverCardFollow() - 测试悬停卡片关注');
console.log('   warmupBot.followUser() - 测试传统关注功能');
console.log('   warmupBot.likeTweet() - 测试点赞功能');
console.log('   warmupBot.retweetTweet() - 测试转发功能');
console.log('   warmupBot.scrollAndBrowse() - 测试浏览功能');

// 页面跳转检测和自动恢复
function checkPageNavigation() {
    const currentUrl = window.location.href;
    const isTwitterHome = currentUrl.includes('twitter.com/home') || currentUrl.includes('x.com/home');
    const isUserProfile = currentUrl.includes('twitter.com/') && currentUrl.includes('/') && !currentUrl.includes('/home');
    
    if (isUserProfile) {
        console.log('🎯 检测到进入用户主页，开始关注流程...');
        console.log('📍 当前页面:', currentUrl);
        
        // 延迟执行，确保页面完全加载
        setTimeout(async () => {
            try {
                await warmupBot.continueFollowProcess();
            } catch (error) {
                console.error('关注流程失败:', error);
                // 尝试返回
                try {
                    window.history.back();
                } catch (e) {
                    console.log('返回失败:', e);
                }
            }
        }, 2000);
    } else if (isTwitterHome) {
        console.log('🏠 检测到返回推特主页，继续预热...');
        // 如果全局脚本还在运行，继续执行
        if (window.twitterBot.isRunning) {
            console.log('✅ 脚本继续运行');
        } else {
            console.log('🔄 重新启动脚本...');
            window.twitterBot.start();
        }
    }
}

// 监听页面变化
window.addEventListener('popstate', checkPageNavigation);
window.addEventListener('pushstate', checkPageNavigation);
window.addEventListener('replacestate', checkPageNavigation);

// 监听URL变化
let currentUrl = window.location.href;
setInterval(() => {
    if (window.location.href !== currentUrl) {
        currentUrl = window.location.href;
        console.log('🔄 检测到URL变化:', currentUrl);
        checkPageNavigation();
    }
}, 1000);

// 页面加载完成后检查
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', checkPageNavigation);
} else {
    checkPageNavigation();
}

// 自动开始预热
warmupBot.startWarmup();
