// 推特养号助手 - 内容脚本
class TwitterBotContent {
    constructor() {
        this.isRunning = false;
        this.bot = null;
        this.config = null;
        this.init();
    }

    init() {
        // 监听来自后台脚本的消息
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            this.handleMessage(message, sender, sendResponse);
            return true;
        });

        // 监听页面变化
        this.observePageChanges();

        console.log('🤖 推特养号助手内容脚本已加载');
    }

    handleMessage(message, sender, sendResponse) {
        console.log('📨 收到消息:', message);
        
        switch (message.action) {
            case 'startBot':
                try {
                    this.startBot(message.config);
                    sendResponse({ success: true });
                } catch (error) {
                    console.error('❌ 启动机器人失败:', error);
                    sendResponse({ success: false, error: error.message });
                }
                break;
                
            case 'stopBot':
                try {
                    this.stopBot();
                    sendResponse({ success: true });
                } catch (error) {
                    console.error('❌ 停止机器人失败:', error);
                    sendResponse({ success: false, error: error.message });
                }
                break;
                
            case 'resumeBot':
                try {
                    this.resumeBot(message.config);
                    sendResponse({ success: true });
                } catch (error) {
                    console.error('❌ 恢复机器人失败:', error);
                    sendResponse({ success: false, error: error.message });
                }
                break;
                
            default:
                console.log('❓ 未知消息类型:', message.action);
                sendResponse({ success: false, message: '未知操作' });
        }
    }

    observePageChanges() {
        // 监听URL变化
        let currentUrl = window.location.href;
        const checkUrlChange = () => {
            if (window.location.href !== currentUrl) {
                currentUrl = window.location.href;
                console.log('🔄 检测到URL变化:', currentUrl);
                this.handlePageNavigation();
            }
        };

        // 每秒检查URL变化
        setInterval(checkUrlChange, 1000);

        // 监听浏览器历史变化
        window.addEventListener('popstate', () => {
            console.log('🔄 检测到浏览器历史变化');
            this.handlePageNavigation();
        });
    }

    handlePageNavigation() {
        if (!this.isRunning) return;

        const currentUrl = window.location.href;
        const isTwitterHome = currentUrl.includes('twitter.com/home') || currentUrl.includes('x.com/home');
        const isUserProfile = currentUrl.includes('twitter.com/') && currentUrl.includes('/') && 
                             !currentUrl.includes('/home') && !currentUrl.includes('/search') && 
                             !currentUrl.includes('/notifications');

        if (isUserProfile) {
            console.log('🎯 检测到进入用户主页，继续关注流程...');
            setTimeout(() => {
                if (this.bot && this.bot.continueFollowProcess) {
                    this.bot.continueFollowProcess();
                }
            }, 2000);
        } else if (isTwitterHome) {
            console.log('🏠 检测到返回推特主页，继续预热...');
            setTimeout(() => {
                if (this.bot && this.bot.resumeWarmup) {
                    this.bot.resumeWarmup();
                }
            }, 1000);
        }
    }

    startBot(config) {
        if (this.isRunning) {
            console.log('⚠️ 机器人已在运行中');
            return;
        }

        this.config = config;
        this.isRunning = true;

        // 创建机器人实例
        this.bot = new TwitterWarmupBot(config);
        
        // 启动机器人
        this.bot.startWarmup();

        console.log('🚀 内容脚本机器人已启动');
    }

    stopBot() {
        if (!this.isRunning) {
            console.log('⚠️ 机器人未在运行');
            return;
        }

        if (this.bot) {
            this.bot.stopWarmup();
        }

        this.isRunning = false;
        this.bot = null;
        console.log('⏹️ 内容脚本机器人已停止');
    }

    resumeBot(config) {
        if (!this.isRunning) {
            console.log('⚠️ 机器人未在运行，无法恢复');
          return;
        }

        this.config = config;
        
        if (this.bot) {
            this.bot.updateConfig(config);
            this.bot.resumeWarmup();
        }

        console.log('🔄 机器人已恢复');
    }
}

// 推特养号机器人核心类
class TwitterWarmupBot {
    constructor(config) {
        this.config = config || {
            tweetsToView: 60,
            likesToGive: 5,
            followsToGive: 3,
            retweetsToGive: 2,
            duration: 20,
            enableLike: true,
            enableFollow: true,
            enableRetweet: true,
            enableScroll: true
        };
        
        this.isRunning = false;
        this.stats = {
            tweetsViewed: 0,
            likesGiven: 0,
            followsGiven: 0,
            retweetsGiven: 0,
            hoverCardsTriggered: 0,
            startTime: Date.now(),
            actionSequence: []
        };
        
        this.actionLog = [];
        this.statusFlushTimer = null;
        this.stopRequested = false;
        this.duration = this.config.duration * 60 * 1000; // 转换为毫秒
        this.targets = {
            tweetsToView: this.config.tweetsToView,
            likesToGive: this.config.likesToGive,
            followsToGive: this.config.followsToGive,
            retweetsToGive: this.config.retweetsToGive
        };
    }

    updateConfig(config) {
        this.config = { ...this.config, ...config };
        this.duration = this.config.duration * 60 * 1000;
        this.targets = {
            tweetsToView: this.config.tweetsToView,
            likesToGive: this.config.likesToGive,
            followsToGive: this.config.followsToGive,
            retweetsToGive: this.config.retweetsToGive
        };
    }

    async waitForCondition(checkFn, timeout = 5000, interval = 200) {
        const start = Date.now();
        while (Date.now() - start < timeout) {
            try {
                if (await checkFn()) {
                    return true;
                }
            } catch (error) {
                // ignore and keep waiting
            }
            await this.delay(interval);
        }
        return false;
    }

    logAction(type, message, status = 'info', extra = {}) {
        const { stage: stageOverride, ...restExtra } = extra || {};

        const entry = {
            id: `${Date.now()}-${Math.random().toString(16).slice(2, 8)}`,
            type,
            message,
            status,
            timestamp: Date.now(),
            ...restExtra
        };
        if (stageOverride) {
            entry.stage = stageOverride;
        }
        this.actionLog.push(entry);
        this.stats.lastAction = entry;
        if (this.actionLog.length > 200) {
            this.actionLog.shift();
        }

        const stage = stageOverride ||
            (this.stopRequested ? 'stopping' : (this.isRunning ? 'progress' : 'stopped'));
        this.scheduleStatusUpdate(stage);
    }

    scheduleStatusUpdate(stage = 'progress') {
        if (this.statusFlushTimer) {
            clearTimeout(this.statusFlushTimer);
        }

        this.statusFlushTimer = setTimeout(() => {
            this.statusFlushTimer = null;
            this.reportStatus(stage);
        }, 150);
    }

    reportStatus(stage = 'progress', extra = {}) {
        try {
            if (!chrome || !chrome.runtime || !chrome.runtime.sendMessage) {
                return;
            }

            if (this.statusFlushTimer) {
                clearTimeout(this.statusFlushTimer);
                this.statusFlushTimer = null;
            }

            const now = Date.now();
            const elapsedMs = Math.max(0, now - (this.stats.startTime || now));
            const remainingMs = Math.max(0, this.duration - elapsedMs);

            const statsSnapshot = {
                tweetsViewed: this.stats.tweetsViewed,
                likesGiven: this.stats.likesGiven,
                followsGiven: this.stats.followsGiven,
                retweetsGiven: this.stats.retweetsGiven,
                hoverCardsTriggered: this.stats.hoverCardsTriggered,
                startTime: this.stats.startTime,
                elapsedSeconds: Math.round(elapsedMs / 1000),
                remainingSeconds: Math.round(remainingMs / 1000),
                totalDurationSeconds: Math.round(this.duration / 1000),
                interactionProgress: this.stats.interactionProgress ?? 0,
                actionSequence: [...(this.stats.actionSequence || [])],
                recentActions: this.actionLog.slice(-20).map(action => ({ ...action })),
                lastAction: this.actionLog.length > 0 ? { ...this.actionLog[this.actionLog.length - 1] } : null,
                stopRequested: this.stopRequested,
                isRunning: this.isRunning,
                stage
            };

            chrome.runtime.sendMessage({
                action: 'botStatus',
                stage,
                isRunning: this.isRunning,
                stats: {
                    ...statsSnapshot,
                    targets: this.targets,
                    durationMinutes: this.config.duration
                },
                ...extra
            }, () => {
                if (chrome.runtime.lastError) {
                    console.warn('⚠️ 状态上报失败:', chrome.runtime.lastError.message);
                }
            });
        } catch (error) {
            console.error('❌ 状态上报失败:', error);
        }
    }

    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    async delayWithCountdown(ms, message) {
        if (!ms || ms <= 0) {
            return;
        }

        let remainingMs = ms;
        console.log(`⏳ ${message} ${Math.ceil(remainingMs / 1000)}秒...`);

        while (remainingMs > 0) {
            if (!this.isRunning) {
                console.log(`⚠️ ${message} 中止，任务已停止`);
                return;
            }
            const tick = Math.min(1000, remainingMs);
            await this.delay(tick);
            remainingMs -= tick;
            this.scheduleStatusUpdate('progress');
            if (remainingMs > 0) {
                console.log(`⏳ ${message} ${Math.ceil(remainingMs / 1000)}秒...`);
            }
        }

        console.log(`✅ ${message} 完成`);
    }

    async simulateHumanClick(element) {
        const rect = element.getBoundingClientRect();
        const x = rect.left + rect.width / 2 + (Math.random() - 0.5) * 10;
        const y = rect.top + rect.height / 2 + (Math.random() - 0.5) * 10;
        
        const mouseMoveEvent = new MouseEvent('mousemove', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(mouseMoveEvent);
        
        await this.delay(100 + Math.random() * 200);
        
        const clickEvent = new MouseEvent('click', {
            clientX: x,
            clientY: y,
            bubbles: true
        });
        element.dispatchEvent(clickEvent);
    }

    isElementInViewport(element) {
        const rect = element.getBoundingClientRect();
        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
            rect.right <= (window.innerWidth || document.documentElement.clientWidth)
        );
    }

    // 自然浏览行为
    async scrollAndBrowse() {
        if (!this.config.enableScroll) return false;
        
        try {
            const scrollDistance = Math.random() * 300 + 300;
            
            window.scrollBy({
                top: scrollDistance,
                behavior: 'smooth'
            });
            
            console.log(`📜 向下滑动 ${Math.round(scrollDistance)}px，浏览内容...`);
            this.stats.tweetsViewed++;
            this.logAction('scroll', `浏览推文 ${this.stats.tweetsViewed}/${this.targets.tweetsToView}`, 'info', {
                scrollDistance: Math.round(scrollDistance)
            });
            
            const isInterested = Math.random() < 0.3;
            
            if (isInterested) {
                const readingTime = Math.random() * 4 + 2;
                console.log(`👀 发现感兴趣的内容！阅读时间: ${readingTime.toFixed(1)}秒`);
                this.logAction('scroll', '发现感兴趣的推文，停留阅读', 'info', {
                    durationSeconds: readingTime.toFixed(1)
                });
                await this.delayWithCountdown(readingTime * 1000, '阅读感兴趣的内容');
            } else {
                const skipTime = Math.random() * 1.5 + 0.5;
                console.log(`⏭️ 内容不感兴趣，快速跳过 (${skipTime.toFixed(1)}秒)`);
                this.logAction('scroll', '当前推文不合适，快速浏览', 'info', {
                    durationSeconds: skipTime.toFixed(1)
                });
                await this.delayWithCountdown(skipTime * 1000, '快速跳过不感兴趣内容');
            }
            
            if (Math.random() < 0.08) {
                const backScroll = 50 + Math.random() * 100;
                window.scrollBy({
                    top: -backScroll,
                    behavior: 'smooth'
                });
                console.log(`↩️ 回看内容 ${Math.round(backScroll)}px`);
                this.logAction('scroll', '回看上一条内容', 'info', {
                    scrollDistance: Math.round(backScroll)
                });
                await this.delayWithCountdown(1000 + Math.random() * 2000, '回看完成，等待继续');
            }
            
            return true;
        } catch (error) {
            console.error('自然浏览失败:', error);
            this.logAction('scroll', `浏览失败: ${error.message || error}`, 'error');
            return false;
        }
    }

    // 点赞推文
    async likeTweet() {
        if (!this.config.enableLike) return false;
        
        try {
            const tweets = Array.from(document.querySelectorAll('article[data-testid="tweet"]'));
            const candidates = tweets
                .map(tweet => {
                    const likeButton = tweet.querySelector('button[data-testid="like"], div[data-testid="like"]');
                    if (!likeButton) return null;

                    const testId = likeButton.getAttribute('data-testid') || '';
                    const aria = (likeButton.getAttribute('aria-label') || '').toLowerCase();
                    const alreadyLiked = testId.includes('unlike') || aria.includes('取消点赞') || aria.includes('liked');
                    if (alreadyLiked) return null;

                    if (likeButton.offsetParent === null) return null;

                    return { tweet, likeButton };
                })
                .filter(Boolean);

            if (!candidates.length) {
                console.log('🔍 未找到可点赞的推文');
                this.logAction('like', '未找到可点赞的推文', 'warning');
                return false;
            }

            const target = candidates[Math.floor(Math.random() * candidates.length)];
            target.tweet.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(600 + Math.random() * 800);

            await this.simulateHumanClick(target.likeButton);
            this.logAction('like', '已点击点赞按钮，等待确认', 'info');

            const liked = await this.waitForCondition(() => {
                const testId = target.likeButton.getAttribute('data-testid') || '';
                const aria = (target.likeButton.getAttribute('aria-label') || '').toLowerCase();
                return testId.includes('unlike') || aria.includes('取消点赞') || aria.includes('liked');
            }, 4000, 200);

            if (liked) {
                console.log('👍 已点赞推文');
                this.stats.likesGiven++;
                this.logAction('like', `点赞成功 (${this.stats.likesGiven}/${this.targets.likesToGive})`, 'success');
                await this.delayWithCountdown(1500 + Math.random() * 2000, '点赞后停留');
                return true;
            }

            this.logAction('like', '点赞似乎未生效，准备重试', 'warning');
            return false;
        } catch (error) {
            console.error('点赞失败:', error);
            this.logAction('like', `点赞失败: ${error.message || error}`, 'error');
            return false;
        }
    }

    // 转发推文
    async retweetTweet() {
        if (!this.config.enableRetweet) return false;
        
        try {
            const retweetSelectors = [
                '[data-testid="retweet"]',
                '[aria-label*="Retweet"]',
                '[aria-label*="转发"]',
                'button[data-testid="retweet"]'
            ];
            
            let retweetButtons = [];
            for (const selector of retweetSelectors) {
                const buttons = document.querySelectorAll(selector);
                retweetButtons = retweetButtons.concat(Array.from(buttons));
            }
            
            const visibleRetweets = retweetButtons.filter(btn => {
                const isVisible = btn.offsetParent !== null;
                const isInViewport = this.isElementInViewport(btn);
                const isNotRetweeted = !btn.closest('[aria-label*="Retweeted"]') && 
                                     !btn.closest('[aria-label*="已转发"]');
                
                return isVisible && isInViewport && isNotRetweeted;
            });
            
            if (visibleRetweets.length === 0) {
                console.log('❌ 未找到可转发的推文');
                this.logAction('retweet', '未找到可转发的推文', 'warning');
                return false;
            }
            
            const randomButton = visibleRetweets[Math.floor(Math.random() * visibleRetweets.length)];
            randomButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(1000 + Math.random() * 1000);
            
            await this.simulateHumanClick(randomButton);
            console.log('🔄 已点击转发按钮');
            
            await this.delay(1500 + Math.random() * 1000);
            
            const confirmSelectors = [
                '[data-testid="retweetConfirm"]',
                '[aria-label*="Retweet"]',
                'button[data-testid="retweetConfirm"]',
                'div[role="button"][data-testid="retweetConfirm"]'
            ];
            
            let confirmButton = null;
            for (const selector of confirmSelectors) {
                const button = document.querySelector(selector);
                if (button && button.offsetParent !== null) {
                    confirmButton = button;
                    break;
                }
            }
            
            if (confirmButton) {
                await this.simulateHumanClick(confirmButton);
                console.log('✅ 已确认转发');
                this.stats.retweetsGiven++;
                this.logAction('retweet', `转发成功 (${this.stats.retweetsGiven}/${this.targets.retweetsToGive})`, 'success');
                
                await this.delayWithCountdown(3000 + Math.random() * 2000, '转发后停留');
                return true;
        } else {
                console.log('⚠️ 未找到确认转发按钮，可能已直接转发');
                this.stats.retweetsGiven++;
                this.logAction('retweet', `转发可能已完成 (${this.stats.retweetsGiven}/${this.targets.retweetsToGive})`, 'info');
                return true;
            }
            
        } catch (error) {
            console.error('转发失败:', error);
            this.logAction('retweet', `转发失败: ${error.message || error}`, 'error');
            return false;
        }
    }

    // 点击头像进入主页关注功能
    async hoverCardFollow() {
        if (!this.config.enableFollow) return false;
        
        const strategies = [
            () => this.followViaInlineTimeline(),
            () => this.followViaRecommendations(),
            () => this.followViaProfileVisit()
        ];

        for (const [index, strategy] of strategies.entries()) {
            try {
                const success = await strategy();
                if (success) {
                    return true;
                }
            } catch (error) {
                console.error('关注策略执行失败:', error);
                this.logAction('follow', `关注策略异常: ${error.message || error}`, 'error');
            }
        }

        this.logAction('follow', '所有关注策略均未成功', 'error');
        return false;
    }

    async followViaInlineTimeline() {
        this.logAction('follow', '策略1：尝试在时间线直接关注', 'info');

        const tweets = Array.from(document.querySelectorAll('article[data-testid="tweet"]')).filter(tweet => tweet.offsetParent !== null);
        const candidates = [];

        for (const tweet of tweets) {
            const button = this.findFollowButtonBySelectors(
                [
                    '[data-testid$="-follow"]',
                    'button[data-testid="follow"]',
                    'div[role="button"][data-testid$="-follow"]'
                ],
                '时间线直接关注',
                undefined,
                tweet
            );
            if (button) {
                candidates.push({ tweet, button });
            }
        }

        if (!candidates.length) {
            this.logAction('follow', '策略1未找到可用的关注按钮', 'warning');
            return false;
        }

        const shuffled = this.shuffleArray(candidates);
        for (const { tweet, button } of shuffled) {
            const username = this.getUsernameFromTweet(tweet) || this.extractUsernameFromHref(button.closest('a')?.getAttribute?.('href'));

            tweet.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(800);

            await this.simulateHumanClick(button);
            this.logAction('follow', '已点击时间线中的关注按钮，等待确认', 'info');

            const success = await this.confirmFollow(button, username);
            if (success) {
                return true;
            }
        }

        this.logAction('follow', '策略1尝试的时间线按钮均未成功', 'warning');
        return false;
    }

    async followViaRecommendations() {
        this.logAction('follow', '策略2：尝试使用推荐关注模块', 'info');

        const userCells = Array.from(document.querySelectorAll('[data-testid="UserCell"]')).filter(cell => cell.offsetParent !== null);
        if (!userCells.length) {
            this.logAction('follow', '策略2未找到推荐关注模块', 'warning');
            return false;
        }

        for (const cell of userCells) {
            const button = this.findFollowButtonBySelectors(
                [
                    'button[data-testid$="-follow"]',
                    'button[data-testid="follow"]',
                    'div[role="button"][data-testid$="-follow"]'
                ],
                '推荐关注模块',
                undefined,
                cell
            );

            if (button) {
                const link = cell.querySelector('a[href^="/"][role="link"]');
                const username = this.extractUsernameFromHref(link?.getAttribute('href'));

                cell.scrollIntoView({ behavior: 'smooth', block: 'center' });
                await this.delay(600);

                await this.simulateHumanClick(button);
                this.logAction('follow', '已点击推荐列表中的关注按钮，等待确认', 'info');

                const success = await this.confirmFollow(button, username);
                if (success) {
                    return true;
                }
            }
        }

        this.logAction('follow', '策略2未成功找到有效关注按钮', 'warning');
        return false;
    }

    async followViaProfileVisit() {
        this.logAction('follow', '策略3：尝试进入用户主页后关注', 'info');

        try {
            const avatarSelectors = [
                '[data-testid="UserAvatar-Container"]',
                '[data-testid="UserAvatar"]',
                '[data-testid="primaryColumn"] [data-testid="UserAvatar-Container"]',
                '[data-testid="primaryColumn"] [data-testid="UserAvatar"]',
                'article [data-testid="UserAvatar-Container"]',
                'article [data-testid="UserAvatar"]',
                'div[role="button"][aria-label*="头像"]',
                'div[role="button"][aria-label*="Avatar"]',
                'div[role="button"][aria-label*="Profile"]',
                '[data-testid="UserCell"] a[role="link"] img'
            ];
            
            let avatarElements = [];
            for (const selector of avatarSelectors) {
                const elements = document.querySelectorAll(selector);
                avatarElements = avatarElements.concat(Array.from(elements));
            }
            
            const basicAvatars = avatarElements.filter(avatar => {
                const isVisible = avatar.offsetParent !== null;
                const hasSize = avatar.offsetWidth > 0 && avatar.offsetHeight > 0;
                return isVisible && hasSize;
            });
            
            const visibleAvatars = basicAvatars.filter(avatar => {
                const isNotSidebar = !avatar.closest('[data-testid="sidebarColumn"]') && 
                                   !avatar.closest('[role="navigation"]') &&
                                   !avatar.closest('nav') &&
                                   !avatar.closest('[data-testid="SideNav"]');
                
                const isInTweetArea = avatar.closest('[data-testid="primaryColumn"]') ||
                                     avatar.closest('[data-testid="tweet"]') ||
                                     avatar.closest('article') ||
                                     avatar.closest('[role="article"]') ||
                                     avatar.closest('[data-testid="UserCell"]');
                
                return isNotSidebar && isInTweetArea;
            });
            
            if (visibleAvatars.length === 0) {
                console.log('❌ 未找到任何头像元素');
                this.logAction('follow', '策略3未找到可点击的用户头像', 'warning');
                return false;
            }
            
            const randomAvatar = visibleAvatars[Math.floor(Math.random() * visibleAvatars.length)];
            randomAvatar.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await this.delay(1000);
            
            const beforeUrl = window.location.href;
            this.logAction('follow', '正在打开某位用户主页', 'info');
            await this.simulateHumanClick(randomAvatar);
            console.log('✅ 已点击头像，等待页面跳转...');
            this.stats.hoverCardsTriggered = (this.stats.hoverCardsTriggered || 0) + 1;
            this.logAction('follow', '已触发进入主页的操作', 'info');
            
            await this.delayWithCountdown(3000, '等待页面跳转');
            
            const afterUrl = window.location.href;
            if (beforeUrl === afterUrl) {
                console.log('⚠️ 页面未跳转，可能点击失败');
                this.logAction('follow', '策略3点击头像后页面未跳转', 'warning');
                return false;
            }
            
            console.log('✅ 成功进入用户主页');
            const username = this.getUsernameFromUrl(afterUrl);
            this.logAction('follow', `成功进入用户主页${username ? ` @${username}` : ''}`, 'info');
            await this.delayWithCountdown(2000, '等待页面完全加载');
            
            return await this.continueFollowProcess();
            
        } catch (error) {
            console.error('点击头像关注失败:', error);
            this.logAction('follow', `策略3进入用户主页失败: ${error.message || error}`, 'error');
            try {
                window.history.back();
                await this.delay(2000);
                this.logAction('follow', '策略3失败后已尝试返回时间线页面', 'info');
            } catch (e) {
                console.log('返回失败:', e);
            }
            return false;
        }
    }

    getUsernameFromTweet(tweet) {
        if (!tweet) return null;
        const link = tweet.querySelector('a[href^="/"][role="link"]');
        if (link) {
            return this.extractUsernameFromHref(link.getAttribute('href'));
        }
        return null;
    }

    getUsernameFromUrl(url = window.location.href) {
        try {
            const parsed = new URL(url, window.location.origin);
            const segments = parsed.pathname.split('/').filter(Boolean);
            if (segments.length > 0) {
                return segments[0].replace('@', '');
            }
        } catch (error) {
            console.warn('⚠️ 解析用户名失败:', error);
        }
        return null;
    }

    extractUsernameFromHref(href) {
        if (!href) return null;
        try {
            const url = new URL(href, window.location.origin);
            const segments = url.pathname.split('/').filter(Boolean);
            if (segments.length > 0) {
                const candidate = segments[0];
                if (candidate && candidate !== 'home' && candidate !== 'notifications') {
                    return candidate.replace('@', '');
                }
            }
        } catch (error) {
            // ignore
        }
        return null;
    }

    shuffleArray(items) {
        const arr = [...items];
        for (let i = arr.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [arr[i], arr[j]] = [arr[j], arr[i]];
        }
        return arr;
    }

    isButtonFollowing(element) {
        if (!element) return false;
        const text = (element.textContent || '').trim().toLowerCase();
        const aria = (element.getAttribute?.('aria-label') || '').trim().toLowerCase();
        const testId = (element.getAttribute?.('data-testid') || '').trim().toLowerCase();

        return text.includes('已关注') ||
               text.includes('following') ||
               aria.includes('已关注') ||
               aria.includes('following') ||
               testId.includes('unfollow');
    }

    isButtonFollowCandidate(element) {
        if (!element) return false;
        if (this.isButtonFollowing(element)) return false;

        const text = (element.textContent || '').trim().toLowerCase();
        const aria = (element.getAttribute?.('aria-label') || '').trim().toLowerCase();
        const testId = (element.getAttribute?.('data-testid') || '').trim().toLowerCase();

        return testId.endsWith('-follow') ||
               testId === 'follow' ||
               text === '关注' ||
               text === 'follow' ||
               text.includes('关注') ||
               text.includes('follow') ||
               aria.includes('关注') ||
               aria.includes('follow');
    }

    async confirmFollow(button, username) {
        const followed = await this.waitForCondition(() => this.isButtonFollowing(button), 5000, 250);
        
        if (followed) {
            this.stats.followsGiven++;
            this.logAction('follow', `关注成功${username ? ` @${username}` : ''} (${this.stats.followsGiven}/${this.targets.followsToGive})`, 'success');
            await this.delayWithCountdown(1500, '关注操作完成');
            return true;
        }

        this.logAction('follow', `关注按钮状态未改变${username ? ` @${username}` : ''}`, 'warning');
        return false;
    }

    findFollowButton() {
        const strategies = [
            () => this.findFollowButtonBySelectors([
                '[data-testid="placementTracking"] button[data-testid$="-follow"]',
                '[data-testid="placementTracking"] button[aria-label*="关注"]',
                '[data-testid="placementTracking"] button[aria-label*="Follow"]'
            ], '置顶模块'),
            () => this.findFollowButtonBySelectors([
                'div[data-testid="primaryColumn"] button[data-testid$="-follow"]',
                'div[data-testid="primaryColumn"] button[data-testid="follow"]'
            ], '主列 follow 浮层按钮'),
            () => this.findFollowButtonBySelectors([
                '[data-testid="primaryColumn"] [data-testid$="-follow"]',
                'main [data-testid="follow"]'
            ], '主列 Follows 结构'),
            () => this.findFollowButtonBySelectors([
                'div[data-testid="primaryColumn"] div[role="button"][data-testid$="-follow"]'
            ], 'div-role follow 按钮'),
            () => this.findFollowButtonBySelectors([
                '[data-testid="placementTracking"] div[role="button"] span',
                'div[data-testid="primaryColumn"] div[role="button"] span'
            ], 'span 文本按钮', element => element.closest('div[role="button"]')),
            () => this.findFollowButtonBySelectors([
                'div[data-testid="primaryColumn"] button',
                'div[data-testid="primaryColumn"] div[role="button"]'
            ], '全部按钮')
        ];

        for (const strategy of strategies) {
            const result = strategy();
            if (result) {
                return result;
            }
        }

        // 最后兜底：在页面任意按钮中查找
        return this.findFollowButtonBySelectors([
            'button',
            'div[role="button"]',
            'a[role="button"]'
        ], '全局按钮');
    }

    findFollowButtonBySelectors(selectors, label = '默认策略', elementResolver, root = document) {
        const seen = new Set();

        const resolveCandidate = (element) => {
            if (!element) return null;
            if (elementResolver) {
                element = elementResolver(element) || element;
            }

            if (!element || seen.has(element)) return null;
            seen.add(element);

            if (element.offsetParent === null) return null;
            if (element.getAttribute && element.getAttribute('disabled') !== null) return null;

            return this.isButtonFollowCandidate(element) ? element : null;
        };

        for (const selector of selectors) {
            try {
                const elements = Array.from(root.querySelectorAll(selector));
                for (const element of elements) {
                    const candidate = resolveCandidate(element);
                    if (candidate) {
                        this.logAction('follow', `使用策略 ${label} 找到关注按钮`, 'info');
                        return candidate;
                    }
                }
            } catch (error) {
                continue;
            }
        }

        return null;
    }

    // 继续关注流程
    async continueFollowProcess() {
        try {
            const currentUrl = window.location.href;
            console.log('📍 当前页面URL:', currentUrl);
            
            console.log('📜 在用户主页浏览内容...');
            const browseCount = 2 + Math.floor(Math.random() * 3);
            
            for (let i = 0; i < browseCount; i++) {
                const scrollDistance = Math.random() * 400 + 200;
                window.scrollBy({
                    top: scrollDistance,
                    behavior: 'smooth'
                });
                console.log(`📜 主页浏览滚动 ${Math.round(scrollDistance)}px (${i + 1}/${browseCount})`);
                
                const readingTime = Math.random() * 2 + 1;
                await this.delayWithCountdown(readingTime * 1000, '浏览用户内容');
            }
            
            console.log('🔍 在用户主页查找关注按钮...');
            this.logAction('follow', '完成主页浏览，正在定位关注按钮', 'info');
            const followButton = this.findFollowButton();
            const username = this.getUsernameFromUrl(currentUrl);
            
            if (followButton) {
                console.log('✅ 在用户主页找到关注按钮');
                followButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
                await this.delay(500);
                
                await this.simulateHumanClick(followButton);
                console.log('✅ 已点击关注按钮');

                const success = await this.confirmFollow(followButton, username);
                
                console.log('🔙 返回上一页...');
                window.history.back();
                await this.delayWithCountdown(2000, '等待返回完成');
                
                console.log('✅ 关注流程完成，已返回');
                this.logAction('follow', '关注流程完成，已返回时间线', 'info');
                return success;
            } else {
                console.log('❌ 在用户主页未找到关注按钮');
                this.logAction('follow', `未找到关注按钮${username ? ` @${username}` : ''}`, 'warning');
                window.history.back();
                await this.delay(1000);
                this.logAction('follow', '未找到关注按钮，已返回时间线', 'warning');
                return false;
            }
            
        } catch (error) {
            console.error('继续关注流程失败:', error);
            this.logAction('follow', `关注流程失败: ${error.message || error}`, 'error');
            try {
                window.history.back();
            } catch (e) {
                console.log('返回失败:', e);
            }
            return false;
        }
    }

    shouldAvoidAction(action) {
        const recentActions = this.stats.actionSequence.slice(-3);
        const sameActionCount = recentActions.filter(a => a === action).length;
        return sameActionCount >= 2;
    }

    shouldForceInteraction() {
        const totalInteractions = this.targets.likesToGive + this.targets.followsToGive + this.targets.retweetsToGive;
        const completedInteractions = this.stats.likesGiven + this.stats.followsGiven + this.stats.retweetsGiven;
        const elapsed = (Date.now() - this.stats.startTime) / 1000 / 60;

        if (totalInteractions === 0) {
            return false;
        }
        
        if (elapsed > 5 && (completedInteractions / totalInteractions) < 0.2) {
            return true;
        }
        
        const recentActions = this.stats.actionSequence.slice(-10);
        if (recentActions.length >= 10 && recentActions.every(action => action === 'scroll')) {
            return true;
        }
        
        return false;
    }

    calculateWaitTime(action) {
        let baseTime = 4000;
        
        switch (action) {
            case 'scroll':
                baseTime = 1500 + Math.random() * 2000;
                break;
            case 'like':
                baseTime = 2000 + Math.random() * 2000;
                break;
            case 'hoverFollow':
            case 'follow':
                baseTime = 4000 + Math.random() * 4000;
                break;
            case 'retweet':
                baseTime = 5000 + Math.random() * 3000;
                break;
        }
        
        const recentActions = this.stats.actionSequence.slice(-2);
        if (recentActions.includes(action)) {
            baseTime *= 1.5;
        }
        
        return Math.round(baseTime);
    }

    selectNextAction() {
        const actions = [];
        
        for (let i = 0; i < 15; i++) {
            actions.push('scroll');
        }
        
        const availableActions = [];
        
        if (this.stats.likesGiven < this.targets.likesToGive && 
            !this.shouldAvoidAction('like') && this.config.enableLike) {
            availableActions.push('like');
        }
        
        if (this.stats.followsGiven < this.targets.followsToGive && 
            this.config.enableFollow) {
            if (!this.shouldAvoidAction('hoverFollow')) {
                availableActions.push('hoverFollow');
            }
        }
        
        if (this.stats.retweetsGiven < this.targets.retweetsToGive && 
            !this.shouldAvoidAction('retweet') && this.config.enableRetweet) {
            availableActions.push('retweet');
        }
        
        const forceInteraction = this.shouldForceInteraction();
        
        if (forceInteraction && availableActions.length > 0) {
            console.log('🚨 强制互动模式激活！');
            availableActions.forEach(action => {
                actions.push(action);
                actions.push(action);
                actions.push(action);
            });
  } else {
            availableActions.forEach(action => {
                let weight = 1;
                
                switch (action) {
                    case 'like':
                        const likeProgress = this.stats.likesGiven / this.targets.likesToGive;
                        if (likeProgress >= 0.8) weight = 0.5;
                        else if (likeProgress >= 0.6) weight = 0.7;
                        break;
                    case 'hoverFollow':
                        const followProgress = this.stats.followsGiven / this.targets.followsToGive;
                        if (followProgress < 0.2) weight = 2;
                        else if (followProgress < 0.5) weight = 1.5;
                        break;
                    case 'retweet':
                        const retweetProgress = this.stats.retweetsGiven / this.targets.retweetsToGive;
                        if (retweetProgress < 0.3) weight = 2;
                        else if (retweetProgress < 0.6) weight = 1.5;
                        break;
                }
                
                const maxInteractions = Math.min(Math.round(weight * 2), 3);
                for (let i = 0; i < maxInteractions; i++) {
                    actions.push(action);
                }
            });
        }
        
        if (this.stats.likesGiven >= this.targets.likesToGive && 
            this.stats.followsGiven >= this.targets.followsToGive && 
            this.stats.retweetsGiven >= this.targets.retweetsToGive) {
            for (let i = 0; i < 8; i++) {
                actions.push('scroll');
            }
        }
        
        const likeProgress = this.stats.likesGiven / this.targets.likesToGive;
        const followProgress = this.stats.followsGiven / this.targets.followsToGive;
        const retweetProgress = this.stats.retweetsGiven / this.targets.retweetsToGive;
        
        let selectedAction;
        
        if (likeProgress >= 0.8 && (followProgress < 0.5 || retweetProgress < 0.5)) {
            const nonLikeActions = actions.filter(action => action !== 'like' && action !== 'scroll');
            if (nonLikeActions.length > 0) {
                selectedAction = nonLikeActions[Math.floor(Math.random() * nonLikeActions.length)];
                console.log(`⚖️ 强制平衡：点赞完成度${Math.round(likeProgress*100)}%，选择其他动作: ${selectedAction}`);
            } else {
                selectedAction = actions[Math.floor(Math.random() * actions.length)];
            }
} else {
            selectedAction = actions[Math.floor(Math.random() * actions.length)];
        }
        
        this.stats.actionSequence = this.stats.actionSequence || [];
        this.stats.actionSequence.push(selectedAction);
        if (this.stats.actionSequence.length > 10) {
            this.stats.actionSequence.shift();
        }
        
        return selectedAction;
    }

    showProgress() {
        const elapsed = Math.round((Date.now() - this.stats.startTime) / 1000);
        const remaining = Math.max(0, Math.round((this.duration - (Date.now() - this.stats.startTime)) / 1000));
        const totalInteractions = this.targets.likesToGive + this.targets.followsToGive + this.targets.retweetsToGive;
        const completedInteractions = this.stats.likesGiven + this.stats.followsGiven + this.stats.retweetsGiven;
        const interactionProgress = totalInteractions > 0 ? Math.round((completedInteractions / totalInteractions) * 100) : 0;
        
        const recentActions = this.stats.actionSequence.slice(-3).join(' → ');
        this.stats.interactionProgress = interactionProgress;
        this.stats.elapsedSeconds = elapsed;
        this.stats.remainingSeconds = remaining;
        
        console.log(`📊 预热进度: ${elapsed}秒 | 推文:${this.stats.tweetsViewed}/${this.targets.tweetsToView} | 点赞:${this.stats.likesGiven}/${this.targets.likesToGive} | 关注:${this.stats.followsGiven}/${this.targets.followsToGive} | 转发:${this.stats.retweetsGiven}/${this.targets.retweetsToGive} | 互动进度:${interactionProgress}% | 最近动作:${recentActions} | 剩余:${remaining}秒`);
    }

    async startWarmup() {
        if (this.isRunning) {
            console.log('⚠️ 预热已在进行中');
            return;
        }
        
        this.isRunning = true;
        this.stopRequested = false;
        this.stats.startTime = Date.now();
        this.logAction('system', '开始推特养号预热流程', 'info', {
            configuration: { ...this.targets, durationMinutes: this.config.duration },
            stage: 'start'
        });
        this.reportStatus('start');

        console.log('🚀 开始推特养号预热（插件版）...');
        console.log(`🎯 目标: 浏览${this.targets.tweetsToView}条推文, 点赞${this.targets.likesToGive}条, 关注${this.targets.followsToGive}个账号, 转发${this.targets.retweetsToGive}条`);
        console.log(`⏰ 预计时长: ${Math.round(this.duration / 60000)}分钟`);
        
        const startTime = this.stats.startTime;
        
        while (Date.now() - startTime < this.duration && this.isRunning) {
            const action = this.selectNextAction();
            
            try {
                switch (action) {
                    case 'scroll':
                        await this.scrollAndBrowse();
                        break;
                    case 'like':
                        await this.likeTweet();
                        break;
                    case 'hoverFollow':
                        await this.hoverCardFollow();
                        break;
                    case 'retweet':
                        await this.retweetTweet();
                        break;
                }
                
                this.showProgress();
                this.reportStatus('progress');
                if (!this.isRunning) {
                    break;
                }

                const elapsedMs = Date.now() - startTime;
                if (elapsedMs >= this.duration) {
                    console.log('⏰ 已达到预设时长，准备结束流程');
                    break;
                }

                const remainingMs = Math.max(0, this.duration - elapsedMs);
                const waitTime = Math.min(this.calculateWaitTime(action), remainingMs);
                await this.delayWithCountdown(waitTime, '浏览预热等待');
                
            } catch (error) {
                console.error('预热动作失败:', error);
                this.logAction('error', `动作执行失败: ${error.message || error}`, 'error');
            }
        }
        
        this.isRunning = false;
        
        if (this.stopRequested) {
            this.showFinalStats(false);
            this.reportStatus('stopped', { stopped: true });
            return;
        }

        const finishedNaturally = Date.now() - startTime >= this.duration;
        this.showFinalStats(finishedNaturally);
        this.reportStatus(finishedNaturally ? 'completed' : 'stopped', finishedNaturally ? { completed: true } : { stopped: true });
    }

    resumeWarmup() {
        if (this.isRunning) {
            console.log('🔄 机器人已恢复运行');
            this.logAction('system', '页面重新加载后继续执行任务', 'info', { stage: 'resume' });
            this.reportStatus('resume');
        } else {
            console.log('⚠️ 机器人未在运行，无法恢复');
            this.logAction('system', '尝试恢复但机器人未运行', 'warning', { stage: 'stopped' });
        }
    }

    showFinalStats(completed = true) {
        const totalTime = Math.round((Date.now() - this.stats.startTime) / 1000);
        this.stats.elapsedSeconds = totalTime;
        this.stats.remainingSeconds = 0;
        
        const statusEmoji = completed ? '✅' : '⏹️';
        const statusText = completed ? '浏览预热完成!' : '浏览预热结束';
        console.log(`${statusEmoji} ${statusText}`);
        if (!completed) {
            console.log('⚠️ 未完全达到预设目标，以下为当前统计：');
        }
        console.log(`📈 最终统计:`);
        console.log(`   总时间: ${totalTime}秒 (${Math.round(totalTime/60)}分钟)`);
        console.log(`   浏览推文: ${this.stats.tweetsViewed}条`);
        console.log(`   点赞: ${this.stats.likesGiven}条`);
        console.log(`   关注: ${this.stats.followsGiven}个账号`);
        console.log(`   转发: ${this.stats.retweetsGiven}条`);
        
        const percent = (value, total) => total > 0 ? Math.round((value / total) * 100) : 0;
        
        const tweetProgress = percent(this.stats.tweetsViewed, this.targets.tweetsToView);
        const likeProgress = percent(this.stats.likesGiven, this.targets.likesToGive);
        const followProgress = percent(this.stats.followsGiven, this.targets.followsToGive);
        const retweetProgress = percent(this.stats.retweetsGiven, this.targets.retweetsToGive);
        
        console.log(`   完成度: 推文${tweetProgress}% | 点赞${likeProgress}% | 关注${followProgress}% | 转发${retweetProgress}%`);
        this.logAction('system', completed ? '预热流程已全部完成' : '预热流程已提前结束', completed ? 'success' : 'warning', {
            totals: {
                tweetsViewed: this.stats.tweetsViewed,
                likesGiven: this.stats.likesGiven,
                followsGiven: this.stats.followsGiven,
                retweetsGiven: this.stats.retweetsGiven
            },
            progress: { tweetProgress, likeProgress, followProgress, retweetProgress },
            stage: completed ? 'completed' : 'stopped'
        });
    }

    stopWarmup() {
        if (!this.isRunning) {
            console.log('⚠️ 预热未在运行，无需停止');
            this.logAction('system', '收到停止请求，但机器人已停止', 'warning', { stage: 'stopped' });
            return;
        }

        this.stopRequested = true;
        this.isRunning = false;
        console.log('⏹️ 正在停止预热流程...');
        this.logAction('system', '正在停止预热流程', 'warning', { stage: 'stopping' });
        this.reportStatus('stopping', { stopped: true });
    }
}

// 创建内容脚本实例
const contentBot = new TwitterBotContent();
