// 推特养号助手 - 后台脚本
class TwitterBotBackground {
    constructor() {
        this.isRunning = false;
        this.tabId = null;
        this.latestStats = null;
        this.config = {
            tweetsToView: 60,
            likesToGive: 5,
            followsToGive: 3,
            retweetsToGive: 2,
            duration: 20, // 分钟
            enableLike: true,
            enableFollow: true,
            enableRetweet: true,
            enableScroll: true
        };
        this.init();
    }

    async init() {
        // 监听来自popup的消息
        chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
            this.handleMessage(message, sender, sendResponse);
            return true; // 保持消息通道开放
        });

        // 监听标签页更新
        chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
            if (changeInfo.status === 'complete' && this.isRunning && this.tabId === tabId) {
                this.handlePageLoad(tabId);
            }
        });

        // 加载配置
        await this.loadConfig();

        console.log('🤖 推特养号助手后台脚本已启动');
    }

    async loadConfig() {
        try {
            const result = await chrome.storage.sync.get([
                'tweetsToView', 'likesToGive', 'followsToGive', 'retweetsToGive',
                'duration', 'enableLike', 'enableFollow', 'enableRetweet', 'enableScroll'
            ]);
            
            this.config = {
                tweetsToView: result.tweetsToView || 60,
                likesToGive: result.likesToGive || 5,
                followsToGive: result.followsToGive || 3,
                retweetsToGive: result.retweetsToGive || 2,
                duration: result.duration || 20,
                enableLike: result.enableLike !== false,
                enableFollow: result.enableFollow !== false,
                enableRetweet: result.enableRetweet !== false,
                enableScroll: result.enableScroll !== false
            };
            
            console.log('📋 配置已加载:', this.config);
        } catch (error) {
            console.error('❌ 加载配置失败:', error);
        }
    }

    async saveConfig(config) {
        try {
            await chrome.storage.sync.set(config);
            this.config = { ...this.config, ...config };
            console.log('💾 配置已保存:', this.config);
        } catch (error) {
            console.error('❌ 保存配置失败:', error);
        }
    }

    async handleMessage(message, sender, sendResponse) {
        switch (message.action) {
            case 'start':
                try {
                    await this.loadConfig();

                    // 获取当前活动标签页
                    let [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

                    if (!tab) {
                        [tab] = await chrome.tabs.query({ active: true, lastFocusedWindow: true });
                    }

                    if (!tab) {
                        const twitterTabs = await chrome.tabs.query({
                            active: true,
                            url: ['https://twitter.com/*', 'https://x.com/*']
                        });
                        tab = twitterTabs[0];
                    }
                    
                    if (!tab) {
                        sendResponse({ success: false, message: '无法获取当前标签页' });
                        return;
                    }

                    if (typeof tab.id !== 'number') {
                        sendResponse({ success: false, message: '标签页信息异常' });
                        return;
                    }
                    
                    if (!tab.url) {
                        sendResponse({ success: false, message: '标签页URL为空' });
                        return;
                    }
                    
                    if (!tab.url.includes('twitter.com') && !tab.url.includes('x.com')) {
                        sendResponse({ success: false, message: '请在推特页面使用此工具' });
                        return;
                    }
                    
                    console.log('📍 当前标签页:', tab.url);
                    this.startBot(tab.id);
                    sendResponse({ success: true, message: '机器人已启动' });
                } catch (error) {
                    console.error('❌ 启动失败:', error);
                    sendResponse({ success: false, message: '启动失败: ' + error.message });
                }
                break;
                
            case 'stop':
                this.stopBot();
                sendResponse({ success: true, message: '机器人已停止' });
                break;
                
            case 'status':
                sendResponse({ 
                    success: true, 
                    isRunning: this.isRunning,
                    config: this.config,
                    stats: this.latestStats
                });
                break;
            
            case 'botStatus':
                console.log('📨 收到机器人状态更新', {
                    stage: message.stage,
                    isRunning: message.isRunning,
                    timestamp: new Date().toISOString()
                });

                this.latestStats = message.stats ? { ...message.stats, stage: message.stage || null } : null;

                if (typeof message.isRunning === 'boolean') {
                    this.isRunning = message.isRunning;
                    if (!this.isRunning) {
                        this.tabId = null;
                    }
                }

                if (this.latestStats) {
                    try {
                        chrome.runtime.sendMessage({
                            action: 'botBroadcast',
                            stats: this.latestStats
                        });
                    } catch (broadcastError) {
                        console.warn('⚠️ 广播状态失败:', broadcastError);
                    }
                }

                sendResponse({ success: true });
                break;
            case 'updateConfig':
                await this.saveConfig(message.config);
                sendResponse({ success: true, message: '配置已更新' });
                break;
                
            case 'getConfig':
                sendResponse({ success: true, config: this.config });
                break;
                
            default:
                sendResponse({ success: false, message: '未知操作' });
        }
    }

    async startBot(tabId) {
        if (this.isRunning) {
            console.log('⚠️ 机器人已在运行中');
            return;
        }

        this.tabId = tabId;
        this.isRunning = true;
        this.latestStats = null;

        try {
            // 注入内容脚本
            await chrome.scripting.executeScript({
                target: { tabId: tabId },
                files: ['content.js']
            });

            // 等待一下确保脚本加载完成
            await new Promise(resolve => setTimeout(resolve, 1000));

            // 发送启动消息到内容脚本
            await chrome.tabs.sendMessage(tabId, {
                action: 'startBot',
                config: this.config
            });

            console.log('🚀 机器人已启动');
        } catch (error) {
            console.error('❌ 启动机器人失败:', error);
            this.isRunning = false;
        }
    }

    async stopBot() {
        if (!this.isRunning) {
            console.log('⚠️ 机器人未在运行');
            return;
        }

        try {
            if (this.tabId) {
                await chrome.tabs.sendMessage(this.tabId, {
                    action: 'stopBot'
                });
            }
        } catch (error) {
            console.error('❌ 停止机器人失败:', error);
        }

        this.isRunning = false;
        this.tabId = null;
        this.latestStats = null;
        console.log('⏹️ 机器人已停止');
    }

    async handlePageLoad(tabId) {
        if (!this.isRunning || this.tabId !== tabId) {
            return;
        }

        try {
            // 检查页面是否在推特
            const tab = await chrome.tabs.get(tabId);
            if (!tab.url.includes('twitter.com') && !tab.url.includes('x.com')) {
                console.log('⚠️ 页面不在推特，停止机器人');
                this.stopBot();
                return;
            }

            // 重新注入脚本
            await chrome.scripting.executeScript({
                target: { tabId: tabId },
                files: ['content.js']
            });

            // 发送恢复消息
            await chrome.tabs.sendMessage(tabId, {
                action: 'resumeBot',
                config: this.config
            });

            console.log('🔄 页面加载完成，机器人已恢复');
        } catch (error) {
            console.error('❌ 处理页面加载失败:', error);
        }
    }
}

// 创建后台脚本实例
const backgroundBot = new TwitterBotBackground();
